/**
 * $Id: AbstractBusinessRule.java,v 1.9 2006/08/10 15:19:11 dec Exp $
 */
package bizrules;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.logging.Logger;

import processingError.ErrorProcessor;
import processingError.Location;
import xmldoc.DocumentError;
import xmldoc.ElementReference;
import xmldoc.SimpleElementReference;
import bizrules.registry.BusinessRuleRegistryAccess;

import com.gsl.logging.LoggerFactory;

/**
 * @author Douglas Clinton
 * @since Jan 9, 2006
 * 
 */
public abstract class AbstractBusinessRule extends AbstractElementProcessor implements BusinessRule, ErrorProcessor {

    protected static final String[] EMPTY_STRING_ARRAY = new String[0];

    private String name;

    /**
     * The contents of this list will be provided to the BusinessRuleRegistry
     * when it asks us for our attachment points.
     * 
     * @associates ElementReference
     */
    private final List<ElementReference> attachmentPoints = new ArrayList<ElementReference>();

    /**
     * Each rule must decide where it wishes to be invoked and implement this
     * method to return a String[] of XPaths corresponding to those points.
     * 
     */
    protected abstract String[] getInvocationPoints();

    protected static final Logger LOGGER = LoggerFactory.getLogger("bizrules");

    public AbstractBusinessRule() {
        setupName();
        setupInvocationPoints();
    }

    public void setBusinessRuleRegistryAccess(final BusinessRuleRegistryAccess access) {
        // do nothing
    }

    /**
     * By default we use the class name of the concrete class that extends us as
     * the rule name. This can be overridden by the subclass using
     * setRuleName().
     * 
     */
    protected void setupName() {
        this.name = this.getClass().getSimpleName();
    }

    protected void setupInvocationPoints() {
        /*
         * In order to create the ElementReferences for the attachment points,
         * we need to combine each of the invocation points with each of the
         * applicable namespaces.
         */
        final String[] attachmentPaths = getInvocationPoints();

        if (attachmentPaths != null) {
            final String[] namespaces = getApplicableNamespaces();

            for (final String namespace : namespaces) {
                for (final String path : attachmentPaths) {
                    getAttachmentPoints().add(SimpleElementReference.makeReference(namespace, path));
                }
            }
        }
    }

    @Override
    public String toString() {
        return name;
    }

    public String getName() {
        return name;
    }

    /**
     * This is protected so that only a subclass can override the default rule
     * name based on the concrete class name.
     * 
     */
    protected void setRuleName(final String name) {
        this.name = name;
    }

    /**
     * Implement the method required by the interface.
     */
    public List<ElementReference> getAttachmentPoints() {
        return attachmentPoints;
    }

    public abstract boolean isActive();

    /**
     * Generic implementation will convert the incoming error to a
     * BusinessRuleError with the name of the rule, the same subcode as the
     * incoming error and appropriate extra information.
     */
    @Override
    public boolean handleError(final DocumentError documentError) {
        final BusinessRuleError newError = raiseError(documentError.getSubCode(), documentError.getValueInError(), documentError
                .getExtraInfo(), documentError.getLocation());
        newError.setChainedError(documentError);
        return true;
    }

    /**
     * Generic implementation will ignore the incoming error
     */
    public boolean ignoreError(final DocumentError documentError) {
        return true;
    }

    protected BusinessRuleError raiseError(final String subCode, final Object objectInError, final Object extraInfo,
            final Location location) {
        assert getDeclaredSubcodes().contains(subCode) : "Undeclared subcode: " + subCode + " in rule class: "
                + getClass().getName();
        final BusinessRuleError businessRuleError = new BusinessRuleError(this, subCode);
        businessRuleError.setExtraInfo(extraInfo);
        businessRuleError.setLocation(location);
        businessRuleError.setValueInError(objectInError);
        businessRuleError.setErrorProcessor(this);

        if (getXPathLocator() != null) {
            businessRuleError.setXPathLocation(getXPathLocationString());
        }

        getErrorCollector().addError(businessRuleError);

        return businessRuleError;
    }

    /**
     * Subclasses can override this if they need to tailor the location string,
     * e.g. by adding attributes.
     * 
     * @return the string representation of the XPathLocation for this rule with
     *         instance counts included (e.g. "/foo[1]/bar[2]").
     */
    protected String getXPathLocationString() {
        return getXPathLocator().toXPathLocatorString();
    }

    protected BusinessRuleError raiseError(final String subCode, final Object objectInError, final Object extraInfo) {
        return raiseError(subCode, objectInError, extraInfo, null);
    }

    protected BusinessRuleError raiseError(final String subCode, final Object objectInError, final Location location) {
        return raiseError(subCode, objectInError, null, location);
    }

    protected BusinessRuleError raiseError(final String subCode, final Location location) {
        return raiseError(subCode, null, location);
    }

    protected BusinessRuleError raiseError(final String subCode, final Object valueInError) {
        return raiseError(subCode, valueInError, null);
    }

    protected BusinessRuleError raiseError(final String subCode) {
        return raiseError(subCode, null);
    }

    private Set<String> declaredSubcodes;

    public Set<String> getDeclaredSubcodes() {
        if (declaredSubcodes == null) {
            declaredSubcodes = new HashSet<String>();
            final String[] subcodes = getSubcodes();
            for (final String element : subcodes) {
                declaredSubcodes.add(element);
            }
            // Always allow suppress through
            declaredSubcodes.add(DocumentError.SUPPRESS);
        }
        return declaredSubcodes;
    }

    public abstract String[] getSubcodes();

    protected String[] combine(final String[] array1, final String[] array2) {
        final String[] dest = new String[array1.length + array2.length];
        System.arraycopy(array1, 0, dest, 0, array1.length);
        System.arraycopy(array2, 0, dest, array1.length, array2.length);
        return dest;
    }
}
