package bizrules.binding;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import xmldoc.DocumentError;
import xmldoc.ElementReference;

import com.gsl.logging.LoggerFactory;
import com.gsl.util.reflection.BasicMethodLocator;
import com.gsl.util.reflection.MethodLocatorCache;

/**
 * $Id: DocumentErrorBinding.java,v 1.4 2006/08/10 15:19:11 dec Exp $
 */
public class DocumentErrorBinding {

    private static final Logger logger = LoggerFactory.getLogger("bizrules.binding");

    private static MethodLocatorCache errorHandlerMethodLocator = new MethodLocatorCache(new BasicMethodLocator(
            new Class[] { DocumentError.class }, Boolean.TYPE));

    private static MethodLocatorCache isActiveLocator = new MethodLocatorCache(new BasicMethodLocator(new Class[] {}, Boolean.TYPE));

    public static void flush(final ClassLoader classloader) {
        errorHandlerMethodLocator.flush(classloader);
    }

    private final Object boundObject;

    /**
     * @directed true
     */
    private final ElementReference reference;

    private final List<Method> errorHandlerMethods = new ArrayList<Method>();

    private Method isActiveMethod;

    public DocumentErrorBinding(final Object boundObj, final ElementReference ref, final String methodName) {
        this.boundObject = boundObj;
        this.reference = ref;

        setupErrorHandlerMethod(methodName);
        setupIsActiveMethod();
    }

    public DocumentErrorBinding(final Object boundObj, final ElementReference ref, final String[] methodNames) {
        this.boundObject = boundObj;
        this.reference = ref;

        for (final String methodName : methodNames) {
            setupErrorHandlerMethod(methodName);
        }
        setupIsActiveMethod();
    }

    private void setupErrorHandlerMethod(final String methodName) {
        final Method errorHandlerMethod = errorHandlerMethodLocator.locateMethod(boundObject.getClass(), methodName);

        if (errorHandlerMethod == null) {
            logger.severe("No method 'public boolean " + methodName + "(DocumentError)' found  on class " + boundObject.getClass());
        } else {
            errorHandlerMethod.setAccessible(true);
            errorHandlerMethods.add(errorHandlerMethod);
        }
    }

    private void setupIsActiveMethod() {
        isActiveMethod = isActiveLocator.locateMethod(boundObject.getClass(), "isActive");
    }

    /**
     * If the bound object has a method <code>boolean isActive()</code> then it
     * will be invoked to determine if the binding is active. If no such method
     * exists on the bound object then this method returns <code>true</code>.
     */
    public boolean isActive() {
        boolean result = true;

        try {
            if (isActiveMethod != null) {
                final Boolean bool = (Boolean) isActiveMethod.invoke(boundObject, new Object[0]);
                result = bool.booleanValue();
            }
        } catch (final InvocationTargetException e) {
            logger.log(Level.SEVERE, e.getTargetException().getMessage(), e.getTargetException());
        } catch (final Exception e) {
            logger.log(Level.SEVERE, e.getMessage(), e);
        }

        return result;
    }

    public boolean invoke(final DocumentError error) {
        boolean result = false;
        try {
            final Object[] args = new Object[] { error };

            for (final Method errorHandlerMethod : errorHandlerMethods) {
                final Boolean bool = (Boolean) errorHandlerMethod.invoke(boundObject, args);
                result = bool.booleanValue();
                if (result) {
                    break;
                }
            }
        } catch (final InvocationTargetException e) {
            logger.log(Level.SEVERE, e.getTargetException().getMessage(), e.getTargetException());
        } catch (final Exception e) {
            logger.log(Level.SEVERE, e.getMessage(), e);
        }
        return result;
    }

    @Override
    public String toString() {
        return "DocumentErrorBinding [Ref='" + reference + "', methods=" + errorHandlerMethods + " on object '" + boundObject
                + "']";
    }

    public ElementReference getReference() {
        return reference;
    }

    public Object getBoundObject() {
        return boundObject;
    }
}