/*
 * $Id: DocumentEventBinding.java,v 1.7 2006/08/10 15:19:11 dec Exp $
 */
package bizrules.binding;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.logging.Level;
import java.util.logging.Logger;

import xmldoc.DocumentEventType;
import xmldoc.ElementReference;
import bizrules.registry.DocumentEventDetail;

import com.gsl.logging.LoggerFactory;
import com.gsl.util.reflection.BasicMethodLocator;
import com.gsl.util.reflection.MethodLocatorCache;

/**
 * Binds a method on an object to a document event so that that method will be
 * called when the event occurs on the given element reference.
 * 
 * The callback method on the bound object must have a signature of the form
 * <em>public void method(DocumentEventDetail)</em>
 * 
 * @see xmldoc.DocumentEventType
 */
public class DocumentEventBinding {

    private static final Logger logger = LoggerFactory.getLogger("coreXML.rules");

    private static MethodLocatorCache eventMethodLocator = new MethodLocatorCache(new BasicMethodLocator(
            new Class[] { DocumentEventDetail.class }, Void.TYPE));

    public static void flush(final ClassLoader classloader) {
        eventMethodLocator.flush(classloader);
    }

    private final Object boundObject;

    /**
     * @directed true
     */

    private final ElementReference reference;

    private final String methodName;

    private Method eventMethod;

    /**
     * @directed true
     */

    private final DocumentEventType eventType;

    /**
     * @param boundObj
     *            this is the object which will be called when the event occurs
     * @param ref
     *            the callback will occur when the event happens at this element
     *            reference
     * @param methodName
     *            this is the method on boundObj which will be called
     * @param eventType
     *            this is the event which will trigger the callback. Valid event
     *            types are declared as constants on the DocumentEventType
     *            interface.
     */
    public DocumentEventBinding(final Object boundObj, final ElementReference ref, final String methodName,
            final DocumentEventType eventType) {
        this.boundObject = boundObj;
        this.reference = ref;
        this.methodName = methodName;
        this.eventType = eventType;

        setupEventMethod();
    }

    private void setupEventMethod() {
        eventMethod = eventMethodLocator.locateMethod(boundObject.getClass(), methodName);

        if (eventMethod == null) {
            logger.severe("No method 'public void " + methodName + "()' found  on class " + boundObject.getClass());
        } else {
            eventMethod.setAccessible(true);
        }
    }

    public void invoke(final DocumentEventDetail detail) {
        try {
            eventMethod.invoke(boundObject, new Object[] { detail });
        } catch (final InvocationTargetException e) {
            logger.log(Level.SEVERE, e.getTargetException().getMessage(), e.getTargetException());
        } catch (final Exception e) {
            logger.log(Level.SEVERE, e.getMessage(), e);
        }
    }

    @Override
    public String toString() {
        return "ElementEventBinding [Ref='" + reference + "', method='public void " + methodName + "()' on object '" + boundObject
                + "' for event '" + eventType + "']";
    }

    public DocumentEventType getEventType() {
        return eventType;
    }

    public ElementReference getReference() {
        return reference;
    }

    @Override
    public int hashCode() {
        return reference.hashCode() ^ methodName.hashCode() ^ boundObject.hashCode() ^ eventType.hashCode();
    }

    @Override
    public boolean equals(final Object other) {
        boolean result = false;

        if (other != null && this.getClass() == other.getClass()) {
            final DocumentEventBinding that = (DocumentEventBinding) other;

            result = this.boundObject.equals(that.boundObject) && this.methodName.equals(that.methodName)
                    && this.reference.equals(that.reference) && (this.eventType == that.eventType);
        }

        return result;
    }
}
