/*
 * $Id: BusinessRuleRegistryImpl.java,v 1.10 2006/08/10 15:19:11 dec Exp $
 */

package bizrules.registry;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;
import java.util.logging.Logger;

import processingError.ProcessingErrorCollector;
import xmldoc.Attribute;
import xmldoc.DocumentError;
import xmldoc.DocumentEventType;
import xmldoc.ElementReference;
import xmldoc.SimpleElementReference;
import xmldoc.sax.XPathLocator;
import bizrules.BusinessRule;
import bizrules.ElementProcessor;
import bizrules.binding.DocumentErrorBinding;
import bizrules.binding.DocumentEventBinding;
import bizrules.binding.DocumentEventListener;
import bizrules.binding.ElementBinding;
import bizrules.binding.ElementValueListener;
import bizrules.params.BizrulesParams;

import com.gsl.logging.LoggerFactory;

/**
 * This class is responsible for taking xmlDoc events (startElement, endElement,
 * endDocument, etc) and dispatching them to the correct processors and rules
 * according to the information that they provide at registration time.
 * 
 * @author Douglas Clinton
 * @since Jan 6, 2006
 * 
 */
public class BusinessRuleRegistryImpl implements BusinessRuleRegistry, BusinessRuleRegistryAccess {

    private final Logger logger = LoggerFactory.getLogger("bizrules.registry");

    private final Map<ElementReference, Collection<ElementBinding>> elementBindings = new HashMap<ElementReference, Collection<ElementBinding>>();

    private final ErrorHandlerRegistry errorHandlerRegistry = new ErrorHandlerRegistry();

    /**
     * Processors and rules will be given a reference to this error collector
     * when they are registered so that they have somewhere to put errors which
     * they raise.
     */
    private final ProcessingErrorCollector errorCollector;

    /**
     * List of namespace handlers.
     */
    private final List<NamespaceHandler> namespaceHandlers = new ArrayList<NamespaceHandler>();

    /**
     * This map resolves a namespace URI to a Map of rules keyed by XPath.
     */
    private final Map<String, Map<String, List<BusinessRule>>> ruleMaps = new HashMap<String, Map<String, List<BusinessRule>>>();

    /**
     * 
     */
    private final Map<DocumentEventType, List<DocumentEventBinding>> eventBindings = new HashMap<DocumentEventType, List<DocumentEventBinding>>();

    /**
     * Holds the reference of the element which we are currently processing.
     */
    private ElementReference currentElementReference;

    /**
     * And here we hold the stack of elements as we progress up and down the XML
     * tree structure.
     */
    private final Stack<ElementReference> elementReferenceStack = new Stack<ElementReference>();

    /**
     * In some environments we only want to log Schema errors or Business rule
     * errors, not both. In these situations we will no invoke any rules after
     * we see a DocumentError.
     */
    private boolean ignoreRulesFollowingDocumentErrors = false;

    private boolean invokeRules = true;

    private XPathLocator xPathLocator = null;

    /**
     * Register a namespace handler.
     */
    public void registerNamespaceHandler(final NamespaceHandler namespaceHandler) {
        namespaceHandlers.add(namespaceHandler);
    }

    /**
     * 
     * @param errorCollector
     *            where errors generated by the rules should be sent.
     */
    public BusinessRuleRegistryImpl(final ProcessingErrorCollector errorCollector, final BizrulesParams params) {
        this.errorCollector = errorCollector;
        this.ignoreRulesFollowingDocumentErrors = params.getIgnoreBusinessRulesFollowingDocumentError();
        this.invokeRules = !params.getIgnoreBusinessRules();
    }

    /**
     * The rule will be interrogated for its attachment points and bindings. It
     * will be given a ProcessingErrorCollector where it should put any errors
     * it generates.
     */
    public void registerBusinessRule(final BusinessRule rule) {
        initializeAttachmentPoints(rule);

        rule.setBusinessRuleRegistryAccess(this);
        registerElementProcessor(rule);
    }

    public void registerElementProcessor(final ElementProcessor processor) {
        initializeReferenceBindings(processor);
        initializeEventBindings(processor);
        initializeErrorBindings(processor);

        processor.setErrorCollector(errorCollector);
        processor.setXPathLocator(getXPathLocator());
        processor.postRegistrationSetup();
    }

    /**
     * Value bindings to objects which are not rules or processors (i.e. java
     * beans) can be registered using this method.
     */
    public void registerValueBinding(final ElementBinding binding) {
        addElementValueBinding(binding);
    }

    private void initializeErrorBindings(final ElementProcessor processor) {
        final List<DocumentErrorBinding> bindings = processor.getErrorBindings();
        for (int i = 0; i < bindings.size(); i++) {
            final DocumentErrorBinding binding = bindings.get(i);
            errorHandlerRegistry.addErrorBinding(binding);
        }
    }

    private void initializeReferenceBindings(final ElementValueListener listener) {
        final List<ElementBinding> bindings = listener.getValueBindings();
        for (int i = 0; i < bindings.size(); i++) {
            final ElementBinding binding = bindings.get(i);
            addElementValueBinding(binding);
        }
    }

    private void initializeEventBindings(final DocumentEventListener listener) {
        final List<DocumentEventBinding> bindings = listener.getEventBindings();
        for (int i = 0; i < bindings.size(); i++) {
            final DocumentEventBinding binding = bindings.get(i);
            addDocumentEventBinding(binding);
        }
    }

    private void addElementValueBinding(final ElementBinding binding) {
        Collection<ElementBinding> bindings = elementBindings.get(binding.getReference());
        if (bindings == null) {
            bindings = new ArrayList<ElementBinding>();
            elementBindings.put(binding.getReference(), bindings);
        }
        bindings.add(binding);

    }

    public void removeElementValueBinding(final Object boundObject, final ElementReference reference, final String boundTo) {
        final Collection<ElementBinding> bindings = lookupValueBindings(reference);
        if (bindings != null) {
            Collection<ElementBinding> toRemove = new ArrayList<ElementBinding>();
            for (ElementBinding binding : bindings) {
                if (binding.getBoundObject() == boundObject && binding.getBoundProperty().equals(boundTo)) {
                    toRemove.add(binding);
                }
            }
            bindings.removeAll(toRemove);
        }
    }

    private void addDocumentEventBinding(final DocumentEventBinding binding) {
        List<DocumentEventBinding> bindings = lookupEventBindings(binding.getEventType());
        if (bindings == null) {
            bindings = new ArrayList<DocumentEventBinding>();
            setDocumentEventBindings(binding, bindings);
        }
        bindings.add(binding);
    }

    public void removeDocumentEventBinding(final Object boundObject, final ElementReference reference, final String boundTo,
            final DocumentEventType eventType) {
        final DocumentEventBinding binding = new DocumentEventBinding(boundObject, reference, boundTo, eventType);
        removeDocumentEventBinding(binding);
    }

    private void removeDocumentEventBinding(final DocumentEventBinding binding) {
        final List<DocumentEventBinding> bindings = new ArrayList<DocumentEventBinding>(lookupEventBindings(binding.getEventType()));
        bindings.remove(binding);
        setDocumentEventBindings(binding, bindings);

        // Check if this was the only binding for the reference. If so, clean up
        // the binding table.
        if (bindings.size() == 0) {
            eventBindings.remove(binding.getEventType());
        }
    }

    private void setDocumentEventBindings(final DocumentEventBinding binding, final List<DocumentEventBinding> bindings) {
        eventBindings.put(binding.getEventType(), bindings);
    }

    private void initializeAttachmentPoints(final BusinessRule rule) {
        final List<ElementReference> attachmentPoints = rule.getAttachmentPoints();
        for (int i = 0; i < attachmentPoints.size(); i++) {
            final ElementReference attachmentPoint = attachmentPoints.get(i);
            final List<BusinessRule> ruleList = lookupRules(attachmentPoint);
            ruleList.add(rule);
        }
    }

    private void invokeRules(final ElementReference elementRef) {
        if (invokeRules) {
            final List<BusinessRule> rules = lookupRules(elementRef);

            for (int i = 0; i < rules.size(); i++) {
                final BusinessRule rule = rules.get(i);
                if (rule.isActive()) {
                    try {
                        rule.processRule();
                    } catch (final Error e) {
                        logger.severe("Caught error processing rule " + rule.getClass().getName());
                        throw e;
                    }
                }
            }
        }
    }

    private List<BusinessRule> lookupRules(final ElementReference attachmentPoint) {
        final Map<String, List<BusinessRule>> ruleMap = lookupRuleMap(attachmentPoint);
        List<BusinessRule> ruleList = ruleMap.get(attachmentPoint.getXPath());
        if (ruleList == null) {
            ruleList = new ArrayList<BusinessRule>();
            ruleMap.put(attachmentPoint.getXPath(), ruleList);
        }
        return ruleList;
    }

    private Map<String, List<BusinessRule>> lookupRuleMap(final ElementReference attachmentPoint) {
        Map<String, List<BusinessRule>> ruleMap = ruleMaps.get(attachmentPoint.getNamespaceURL());
        if (ruleMap == null) {
            ruleMap = new HashMap<String, List<BusinessRule>>();
            ruleMaps.put(attachmentPoint.getNamespaceURL(), ruleMap);
        }
        return ruleMap;
    }

    private Collection<ElementBinding> lookupValueBindings(final ElementReference elementRef) {
        return elementBindings.get(elementRef);
    }

    private List<DocumentEventBinding> lookupEventBindings(final DocumentEventType eventType) {
        return eventBindings.get(eventType);
    }

    public void startNamespace(final String namespace) {
        for (final NamespaceHandler namespaceHandler : namespaceHandlers) {
            namespaceHandler.handleNamespace(namespace);
        }
    }

    public void endNamespace(final String namespace) {
        // no-op
    }

    public void startElement(final ElementReference elementRef, final Attribute[] attributes) {
        final ElementReference parentElementReference = currentElementReference;
        elementReferenceStack.push(currentElementReference);
        currentElementReference = elementRef;

        final Collection<ElementBinding> bindings = lookupValueBindings(elementRef);

        if (bindings != null) {
            for (ElementBinding binding : bindings) {
                binding.startElement(attributes);
            }

        }

        dispatchEvents(elementRef, DocumentEventType.startElement, new StartElementDetail(parentElementReference, elementRef,
                attributes));
    }

    public void endElement(final ElementReference elementRef) {

        final Collection<ElementBinding> bindings = lookupValueBindings(elementRef);
        if (bindings != null) {
            // Make sure we only generate a single copy of the element text
            // to be handed to all the bindings so that we don't generate
            // multiple copies of a very large value.
            StringBuilder b = elementValues.get(elementRef);
            String v = "";
            if (b != null) {
                v = b.toString();
                // and release the StringBuilder as it's no longer needed
                elementValues.remove(elementRef);
                b = null;
            }
            for (ElementBinding binding : bindings) {
                binding.setElementText(v);

                binding.endElement();
                binding.reset();
            }
        }

        dispatchEvents(elementRef, DocumentEventType.endElement, new EndElementDetail(elementRef));

        invokeRules(elementRef);

        currentElementReference = elementReferenceStack.pop();
    }

    private void dispatchEvents(final ElementReference elementRef, final DocumentEventType documentEventType,
            final DocumentEventDetail eventDetail) {
        final List<DocumentEventBinding> bindings = lookupEventBindings(documentEventType);
        if (bindings != null) {
            for (int i = 0; i < bindings.size(); i++) {
                final DocumentEventBinding binding = bindings.get(i);
                if (binding.getReference() == elementRef) {
                    binding.invoke(eventDetail);
                }
            }
        }
    }

    private final Map<ElementReference, StringBuilder> elementValues = new HashMap<ElementReference, StringBuilder>();

    public void elementText(final ElementReference elementRef, final char[] characters, final int start, final int length) {
        final Collection<ElementBinding> bindings = lookupValueBindings(elementRef);
        if (bindings != null) {
            boolean wantsValue = false;
            for (ElementBinding binding : bindings) {
                if (binding.wantsValue()) {
                    wantsValue = true;
                    break;
                }
            }
            if (wantsValue) {
                StringBuilder b = elementValues.get(elementRef);
                if (b == null) {
                    b = new StringBuilder();
                    elementValues.put(elementRef, b);
                }

                b.append(characters, start, length);
            }
        }
    }

    public void startDocument() {
        final List<DocumentEventBinding> bindings = lookupEventBindings(DocumentEventType.startDocument);
        if (bindings != null) {
            for (int i = 0; i < bindings.size(); i++) {
                final DocumentEventBinding binding = bindings.get(i);
                binding.invoke(NullEventDetail.instance());
            }
        }

        startElement(SimpleElementReference.makeReference("/"), new Attribute[] {});
    }

    public void endDocument() {
        endElement(SimpleElementReference.makeReference("/"));

        final List<DocumentEventBinding> bindings = lookupEventBindings(DocumentEventType.endDocument);
        if (bindings != null) {
            for (int i = 0; i < bindings.size(); i++) {
                final DocumentEventBinding binding = bindings.get(i);
                binding.invoke(NullEventDetail.instance());
            }
        }
    }

    /**
     * @return true if there was an error handler registered to handle the
     *         DocumentError and that handler was able to deal with the error.
     *         In this case no DocumentError will be recorded.
     */
    public boolean handleError(final DocumentError error) {
        final List<DocumentErrorBinding> bindings = errorHandlerRegistry.lookupErrorBindings(error.getElementReference());
        // lookupErrorBindings should never return null
        assert bindings != null;

        boolean errorHandledByRule = false;
        for (int i = 0; i < bindings.size() && errorHandledByRule == false; i++) {
            final DocumentErrorBinding binding = bindings.get(i);
            errorHandledByRule = binding.invoke(error);
        }

        /*
         * XXX Bug 28 would require that we remove this, but I'm concerned that
         * it may be essential for the schematron environment to ensure we don't
         * get exceptions thrown out of business rules due to bad submission
         * data. One solution would be to disable this flag (i.e. always
         * continue with business rules) if tolerances are switched on but it's
         * not trivial to get those two pieces of information together at the
         * right time.
         */
        if (ignoreRulesFollowingDocumentErrors) {
            /*
             * Disable invocation of future rules. The environment we're in has
             * said that we don't want to mix Schema and Business errors. We
             * rely on the environment setup to filter out any earlier Business
             * Rule errors.
             */
            invokeRules = false;
        }

        return errorHandledByRule;
    }

    public boolean getIgnoreRulesFollowingDocumentErrors() {
        return ignoreRulesFollowingDocumentErrors;
    }

    public void setIgnoreRulesFollowingDocumentErrors(final boolean ignoreRulesFollowingDocumentErrors) {
        this.ignoreRulesFollowingDocumentErrors = ignoreRulesFollowingDocumentErrors;
    }

    public void setXPathLocator(final XPathLocator locator) {
        this.xPathLocator = locator;
    }

    public XPathLocator getXPathLocator() {
        return xPathLocator;
    }
}
