package com.gsl.docValidator;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Level;

import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import xmldoc.util.AbstractEntityResolver;
import xmldoc.util.EntityResolver;

/**
 * Composites multiple EntityResolver instances behind a single EntityResolver
 * interface. This class is used to allow multiple validator providers to
 * contribute the entity resolution data for just their namespaces, without
 * having to know about other providers or namespaces.
 * <p>
 * Before the parse is started, the framework queries each known provider for a
 * suitable EntityResolver. It then composites these together; an instance of
 * this class is then provided to Xerces to perform the actual entity resoltion.
 * <p>
 * 
 * @author jesse
 */
public class CompositeEntityResolver extends AbstractEntityResolver {

    private final List<EntityResolver> resolvers = new ArrayList<EntityResolver>();

    public CompositeEntityResolver() {
    }

    public void addResolver(final EntityResolver resolver) {
        resolvers.add(resolver);
    }

    public InputSource resolveEntity(final String publicId, final String systemId) throws SAXException, IOException {
        InputSource resolvedInputSource = null;

        if (systemId == null) {
            getLogger().log(Level.SEVERE, "Null systemId is invalid (publicId = '" + publicId + "')");
        }

        final Iterator<EntityResolver> iter = resolvers.iterator();

        while ((resolvedInputSource == null) && iter.hasNext()) {
            final EntityResolver resolver = iter.next();
            resolvedInputSource = resolver.resolveEntity(publicId, systemId);
        }

        if (resolvedInputSource == null) {
            // ignore warning when systemId ends with .dtd - this suppresses
            // warnings for XMLSchema.dtd specifically,
            // and will ignore any other dtds that try to get resolved (can't
            // think of any other examples right now...)
            if (!((systemId != null) && systemId.endsWith(".dtd"))) {
                getLogger().warning("Could not resolve local entity for systemId = " + systemId);
            }
        }

        return resolvedInputSource;
    }

    @Override
    public String getExternalSchemaLocations() {
        final StringBuffer buf = new StringBuffer(512);

        for (final EntityResolver resolver : resolvers) {
            buf.append(resolver.getExternalSchemaLocations());
        }

        return buf.toString();
    }
}
