package com.gsl.docValidator;

import java.util.Date;

/**
 * This class is used to manage the current date/time that any given validation
 * is run with respect to. Normally this will be the current system time, but
 * this class provides a mechanism through which the client can specify the
 * date/time to use.
 * <p>
 * Other framework or validator implementation code (eg. rules or mappers) that
 * need to access the 'current date/time' should call the static
 * {@link #getCurrentDate()} method on this class.
 * <p>
 * This class also contains some methods and constants to pin the current date
 * to any given date. This should <b>ONLY</b> be used for implementing tests.
 * It should be used in production.
 * 
 * @author jesse
 */
public class CurrentDate {

    private static final ThreadLocal<Date> local = new ThreadLocal<Date>();

    /**
     * Returns the current date/time. This method should be used by any code
     * that needs to access the current date/time, rather than relying directly
     * on the system time.
     * 
     * @see ValidationParams#setCurrentDate(Date)
     * 
     * @return the "current date" set by the application, or the current system
     *         time (<code>new Date()</code>) if the "current date" has not
     *         been set.
     */
    public static Date getCurrentDate() {
        Date date = local.get();
        if (date == null) {
            date = new Date();
        }
        return date;
    }

    /**
     * Set the "current date". If this is not null then calls to
     * {@link #getCurrentDate()} will return this value.
     */
    static void init(final Date date) {
        if (date != null) {
            local.set(date);
        }
    }

    /**
     * Reset the system so that {@link #getCurrentDate()} will return the
     * current system date/time.
     */
    static void destroy() {
        local.set(null);
    }

    /**
     * Test code should call this method to initialise the current date to the
     * current system time.
     * <p>
     * <b>THIS METHOD SHOULD NEVER BE CALLED BY PRODUCTION CODE</b>
     */
    public static void initTestMode() {
        initTestMode(null);
    }

    /**
     * Test code should call this method to initialise the current date to the
     * given date.
     * <p>
     * <b>THIS METHOD SHOULD NEVER BE CALLED BY PRODUCTION CODE</b>
     */
    public static void initTestMode(final Date date) {
        init(date);
    }
}
