package com.gsl.docValidator;

/**
 * This is the base class for all Document Validator implementations. Subclass
 * this to create a validator implementation that clients can use to validate a
 * given type of input document.
 * <p>
 * Subclasses can call getController() to get a handle on the
 * DocumentValidatorController instance for this validator. This controller is
 * the entry point to the document validator framework.
 * <p>
 * Subclasses may also override the getServices() method to return an
 * alternative DocumentValidatorServices implementation in order to customize
 * the behaviour of the validator.
 * 
 * @see DocumentValidatorController
 * @see DocumentValidatorServices
 * 
 * @author jesse
 */
public abstract class DocumentValidator {

    // internally we reference DocumentValidatorImpl, which is the
    // (package-private) implementation class responsible for co-ordinating most
    // of the validation framework's functionality
    private final DocumentValidatorImpl controller;

    /**
     * Constructs the DocumentValidator.
     */
    protected DocumentValidator() {
        controller = new DocumentValidatorImpl();
    }

    /**
     * Clients may call this method to interrupt the validator.
     * <p>
     * This method interrupts this validator as soon as possible. This method
     * returns immediately, while the thread running the validation will return
     * as soon as possible after the interrupt request has taken effect.
     * <p>
     * This method does nothing if the validator is idle at the time this method
     * is called.
     * <p>
     * Note that this method may have no effect if the validator is interrupted
     * too soon after being started, due to a race condition in the underlying
     * implementation.
     * 
     * @return true if the validator was interrupted, false otherwise
     */
    public boolean interrupt() {
        return controller.interrupt();
    }

    /**
     * This is a callback method that subclasses may wish to override. It is
     * invoked by the framework after all validation has completed and the
     * validation result has been created.
     * <p>
     * By default this method does nothing.
     */
    protected void validationComplete() {
    }

    /**
     * This method should be used by subclasses to get a handle on the
     * DocumentValidatorController instance for this validator.
     * <p>
     * This method is not generally intended to be overridden.
     * 
     * @return The DocumentValidatorController instance for this validator.
     */
    protected DocumentValidatorController getController() {
        return controller;
    }

    /**
     * This method is called by the framework to create an instance of
     * DocumentValidatorServices. This method may be overridden as a way of
     * customising the validator service components for a particular document
     * validator implementation (ie. by subclassing DocumentValidatorServices).
     * <p>
     * By default this method just returns a new instance of
     * DocumentValidatorServices.
     */
    protected DocumentValidatorServices createServices(final ValidationParams params) {
        return new DocumentValidatorServices(params);
    }
}
