package com.gsl.docValidator;

import java.io.InputStream;
import java.io.Reader;

import xmldoc.util.EntityResolver;

/**
 * This interface provides methods that allow a DocumentValidator implementation
 * to configure and invoke the validation framework to validate an input
 * document and obtain the validation result.
 * 
 * @author jesse
 */
public interface DocumentValidatorController {

    /**
     * Implementations may choose to call this to fail faster than normal in the
     * case where the client re-uses validator instances (since this is not
     * allowed). Even if the user does not call this, it is eventually called by
     * the framework prior to invoking the validation.
     */
    void checkOneShotUse();

    /**
     * Before calling validate(), a DocumentValidator implementation should add
     * the necessary ValidatorProvider instances to handle the expected input
     * documents. This method can be used to add a provider.
     * 
     * @param provider
     *            the ValidatorProvider instance to add
     */
    void addProvider(ValidatorProvider provider);

    /**
     * Before calling validate() a DocumentValidator implementation can add any
     * additional namespace resolvers that it needs using this method.
     */
    void addResolver(EntityResolver resolver);

    /**
     * This method is the actual entry-point to the validation framework.
     * DocumentValidator implementations should invoke this method (passing
     * themselves as the first parameter) in order to start the validation of a
     * given input document.
     * 
     * @param validator
     *            The concrete DocumentValidator instance calling this method.
     *            Implementations should generally pass 'this' for this
     *            parameter.
     * @param input
     *            A Reader instance from which to read the input XML document.
     * @param params
     *            The parameters object to use when validating the input
     *            document. May be null, in which case some defaults will be
     *            used.
     * @param delegate
     *            The PostValidationDelegate instance to invoke once all the
     *            errors have been collected for this input document. This may
     *            be null, in which case the framework will provide a default
     *            delegate that will result in an instance of
     *            ValidationResultBean being returned from this method.
     * 
     * @return The object containing the validation result. Note that the
     *         framework delegates to the PostValidationDelegate to actually
     *         create and populate this object.
     * 
     * @throws ValidationException
     *             If there was an problem during validation.
     */
    ValidationResultInterface validate(DocumentValidator validator, Reader input, ValidationParams params,
            PostValidationDelegate delegate) throws ValidationException;

    /**
     * This version takes an InputStream instead of a Reader
     * 
     * @param validator
     *            The concrete DocumentValidator instance calling this method.
     *            Implementations should generally pass 'this' for this
     *            parameter.
     * @param input
     *            An InputStream from which to read the input XML document.
     * @param params
     *            The parameters object to use when validating the input
     *            document. May be null, in which case some defaults will be
     *            used.
     * @param delegate
     *            The PostValidationDelegate instance to invoke once all the
     *            errors have been collected for this input document. This may
     *            be null, in which case the framework will provide a default
     *            delegate that will result in an instance of
     *            ValidationResultBean being returned from this method.
     * 
     * @return The object containing the validation result. Note that the
     *         framework delegates to the PostValidationDelegate to actually
     *         create and populate this object.
     * 
     * @throws ValidationException
     *             If there was an problem during validation.
     */
    ValidationResultInterface validate(DocumentValidator validator, InputStream input, ValidationParams params,
            PostValidationDelegate delegate) throws ValidationException;
}
