package com.gsl.docValidator;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.util.ArrayList;
import java.util.List;

import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

import processingError.ErrorLimitExceeded;
import processingError.ProcessingError;
import xmldoc.DocumentErrorBean;
import xmldoc.sax.DocumentHandler;
import xmldoc.sax.TerminateParse;
import xmldoc.util.EntityResolver;
import bizrules.BusinessRule;
import bizrules.ElementProcessor;
import bizrules.registry.BusinessRuleRegistry;
import bizrules.registry.NamespaceHandler;
import bizrules.util.BusinessRuleCollector;

import com.gsl.sax.xerces.XercesParserMaker;
import com.gsl.util.OneShotUseException;
import com.gsl.util.OneShotUseSupport;

/**
 * This package-private class is responsible for co-ordinating most of the
 * validation framework's functionality. In particular it provides
 * implementations for the framework's DocumentValidatorController and
 * DocumentValidatorImplementor interfaces.
 * <p>
 * 
 * It also provides a default implementation of PostValidationDelegate. This is
 * only used if the provided delegate is null when the parse completes.
 * <p>
 * 
 * @see DocumentValidatorController
 * @see DocumentValidatorImplementor
 * @see PostValidationDelegate
 * 
 * @author jesse
 */
class DocumentValidatorImpl extends OneShotUseSupport implements DocumentValidatorController, DocumentValidatorImplementor,
        PostValidationDelegate, NamespaceHandler {

    private final List<ValidatorProvider> providers = new ArrayList<ValidatorProvider>();

    private final List<EntityResolver> resolvers = new ArrayList<EntityResolver>();

    private ValidationParams params;

    private PostValidationDelegate delegate;

    private BusinessRuleRegistry ruleRegistry;

    private DocumentHandler docHandler;

    private boolean checkOneShotUse = true;

    /**
     * @see DocumentValidatorController#checkOneShotUse()
     */
    @Override
    public void checkOneShotUse() throws OneShotUseException {
        super.checkOneShotUse();
        checkOneShotUse = false;
    }

    @Override
    protected String getOneShotUseExceptionMessage() {
        return "Validator instances can only be used once.  Please create a new validator instance for each document you want to validate.";
    }

    /**
     * @see DocumentValidatorController#addProvider(ValidatorProvider)
     */
    public void addProvider(final ValidatorProvider provider) {
        providers.add(provider);
    }

    public void addResolver(final EntityResolver resolver) {
        resolvers.add(resolver);
    }

    /**
     * @see DocumentValidatorImplementor#getParams()
     */
    public ValidationParams getParams() {
        return params;
    }

    /**
     * @see DocumentValidatorImplementor#getDelegate()
     */
    public PostValidationDelegate getDelegate() {
        return delegate;
    }

    /**
     * @see DocumentValidatorImplementor#setDelegate(PostValidationDelegate)
     */
    public void setDelegate(final PostValidationDelegate delegate) {
        this.delegate = delegate;
    }

    /**
     * @see DocumentValidatorImplementor#addBusinessRules(String)
     */
    public void addBusinessRules(final String basePackageName) {
        addBusinessRules(BusinessRuleCollector.collectRules(basePackageName, getParams().getBizruleParams()));
    }

    /**
     * @see DocumentValidatorImplementor#addBusinessRules(List)
     */
    public void addBusinessRules(final List<BusinessRule> rules) {
        for (final BusinessRule rule : rules) {
            addBusinessRule(rule);
        }
    }

    /**
     * @see DocumentValidatorImplementor#addBusinessRule(BusinessRule)
     */
    public void addBusinessRule(final BusinessRule rule) {
        ruleRegistry.registerBusinessRule(rule);
    }

    /**
     * @see DocumentValidatorImplementor#addElementProcessor(ElementProcessor)
     */
    public void addElementProcessor(final ElementProcessor processor) {
        ruleRegistry.registerElementProcessor(processor);
    }

    /**
     * @see DocumentValidatorController#validate(DocumentValidator, Reader,
     *      ValidationParams, PostValidationDelegate)
     */
    public ValidationResultInterface validate(final DocumentValidator validator, final Reader input, final ValidationParams params,
            final PostValidationDelegate delegate) throws ValidationException {
        return validate(validator, (Object) input, params, delegate);
    }

    /**
     * @see DocumentValidatorController#validate(DocumentValidator, InputStream,
     *      ValidationParams, PostValidationDelegate)
     */
    public ValidationResultInterface validate(final DocumentValidator validator, final InputStream input,
            final ValidationParams params, final PostValidationDelegate delegate) throws ValidationException {
        return validate(validator, (Object) input, params, delegate);
    }

    //
    // this is a private method which deals with the input as an {@code Object},
    // since it can be
    // either an {@code InputStream} or a {@code Reader}.
    //
    private ValidationResultInterface validate(final DocumentValidator validator, final Object input, ValidationParams params,
            PostValidationDelegate delegate) throws ValidationException {

        // ensure validator instances are only used once
        if (checkOneShotUse) {
            checkOneShotUse();
        }
        checkOneShotUse = true;

        try {
            final long startTime = System.currentTimeMillis();

            if (params == null) {
                params = new ValidationParams() {
                };
            }

            CurrentDate.init(params.getCurrentDate());

            this.params = params;
            this.delegate = delegate;

            final CompositeEntityResolver entityResolver = createEntityResolver();
            final DocumentValidatorServices services = setupServices(validator, entityResolver);

            initProviders();

            parseDocument(params, entityResolver, services, input);

            destroyProviders();

            final List<ProcessingError> processingErrors = services.getErrorCollector().getErrors();

            if (delegate == null) {
                delegate = this;
            }
            final ValidationResultInterface result = delegate.createResult(processingErrors, params);

            result.setProcessingErrors(processingErrors);
            result.setProcessingTime(System.currentTimeMillis() - startTime);

            return result;
        } catch (final Exception e) {
            throw new ValidationException(e);
        } finally {
            this.params = null;
            this.delegate = null;
            this.ruleRegistry = null;
            this.docHandler = null;
            this.providers.clear();
            CurrentDate.destroy();
            validator.validationComplete();
        }
    }

    private void parseDocument(final ValidationParams params, final CompositeEntityResolver entityResolver,
            final DocumentValidatorServices services, final Object input) throws IOException, SAXException {
        XMLReader parser = null;
        try {
            // create parser instance
            parser = createParser(params, entityResolver);

            // create input source
            InputSource inputSource;
            if (input instanceof Reader) {
                inputSource = services.createInputSource((Reader) input);
            } else {
                inputSource = services.createInputSource((InputStream) input);
            }

            // start the parse
            parser.parse(inputSource);
        } catch (final ErrorLimitExceeded ele) {
            if (params.getReportErrorLimitExceeded()) {
                final DocumentErrorBean error = new DocumentErrorBean(ele.getLastError().getLocation());
                error.setSubCode("5012");
                error.setSAXErrorMessage("More than " + ele.getMaxErrors() + " errors generated. Aborting document check.");

                // Can't use the addError() method of the error collector
                // because it will throw the same ErrorLimitExceeded exception
                // that we are dealing with. Instead, add directly to the error
                // list.
                services.getErrorCollector().getErrors().add(error);
            }
        } catch (final TerminateParse tp) {
            // terminated
        } catch (final SAXException e) {
            docHandler.endDocument();
        } finally {
            if (parser != null) {
                XercesParserMaker.release(parser);
            }
        }
    }

    private XMLReader createParser(final ValidationParams params, final CompositeEntityResolver entityResolver) throws SAXException {
        final Object grammarPoolKey = params.getVariant();

        final XMLReader parser = XercesParserMaker.makeParser(params.getParserParams(),
                entityResolver.getExternalSchemaLocations(), grammarPoolKey);
        parser.setContentHandler(docHandler);
        parser.setErrorHandler(docHandler);
        parser.setEntityResolver(docHandler);

        return parser;
    }

    private DocumentValidatorServices setupServices(final DocumentValidator validator, final CompositeEntityResolver entityResolver) {
        final DocumentValidatorServices services = validator.createServices(params);

        ruleRegistry = services.getRuleRegistry();
        ruleRegistry.registerNamespaceHandler(this);

        docHandler = services.getDocumentHandler();
        docHandler.setEntityResolver(entityResolver);
        docHandler.setExceptionDecoder(services.getExceptionDecoder());
        docHandler.setNamespacePrefixMapper(services.getNamespacePrefixMapper());
        docHandler.setErrorCollector(services.getErrorCollector());

        return services;
    }

    boolean interrupt() {
        final DocumentHandler docHandler = this.docHandler;
        if (docHandler != null) {
            docHandler.interrupt();
            return true;
        }
        return false;
    }

    private void initProviders() {
        for (final ValidatorProvider provider : providers) {
            provider.init(this);
        }
    }

    private void destroyProviders() {
        for (final ValidatorProvider provider : providers) {
            provider.destroy();
        }
    }

    private CompositeEntityResolver createEntityResolver() {
        final CompositeEntityResolver compositeResolver = new CompositeEntityResolver();

        for (final ValidatorProvider provider : providers) {
            final EntityResolver resolver = (provider).getEntityResolver(params);
            if (resolver != null) {
                compositeResolver.addResolver(resolver);
            }
        }

        for (final EntityResolver resolver : resolvers) {
            compositeResolver.addResolver(resolver);
        }

        return compositeResolver;
    }

    public void handleNamespace(final String namespace) {
        // provision the namespace
        for (final ValidatorProvider provider : providers) {
            provider.provisionNamespace(namespace, this);
        }
    }

    /**
     * Default implementation of
     * {@link PostValidationDelegate#createResult(List, DocumentValidatorImplementor)}.
     * This is only used if the provided delegate is null when the parse
     * completes. Returns an instance of ValidationResultBean.
     */
    public ValidationResultInterface createResult(final List<ProcessingError> processingErrors, final ValidationParams params) {
        return new ValidationResultBean();
    }
}
