package com.gsl.docValidator;

import java.util.List;

import bizrules.BusinessRule;
import bizrules.ElementProcessor;

/**
 * This class represents the validator during the validation process. A
 * ValidatorProvider uses this interface to provide the necessary configuration
 * to the validator when it provisions its namespace(s). For example, adding
 * business rules, configuring the PostValidationDelegate instance, etc.
 * <p>
 * A PostValidationDelegate also uses this interface, though with some caveats
 * (eg. adding business rules makes no sense for a delegate, since the parse is
 * complete before it gets invoked).
 * 
 * @author jesse
 */
public interface DocumentValidatorImplementor {

    /**
     * Provides access to the validation parameters object for the current
     * validation.
     * 
     * @return The validation parameters object for the current validation.
     */
    ValidationParams getParams();

    /**
     * Returns the PostValidationDelagate instance that will be invoked once the
     * parse has completed.
     * 
     * @return The delegate instance for the current validation.
     */
    PostValidationDelegate getDelegate();

    /**
     * Sets the PostValidationDelagate instance that will be invoked once the
     * parse has completed.
     * <p>
     * In most cases this will not be necessary, since an initial instance can
     * be passed in at the time the validator is started.
     * 
     * @param delegate
     *            The delegate to set.
     */
    void setDelegate(PostValidationDelegate delegate);

    /**
     * Add business rules to the validator by specifying a base package name.
     * 
     * @param basePackageName
     *            The name of the base package containing the rules to add.
     */
    void addBusinessRules(String basePackageName);

    /**
     * Add a list of business rules to the validator.
     * 
     * @param rules
     *            The list of business rules to add.
     */
    void addBusinessRules(List<BusinessRule> rules);

    /**
     * Add a business rule to the validator.
     * 
     * @param rule
     *            The rule to add.
     */
    void addBusinessRule(BusinessRule rule);

    /**
     * Add an element processor to the validator.
     * 
     * @param processor
     *            The element processor to add.
     */
    void addElementProcessor(ElementProcessor processor);
}
