package com.gsl.docValidator;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;

import org.xml.sax.InputSource;

import processingError.ProcessingErrorCollector;
import processingError.SimpleErrorCollector;
import xmldoc.NamespacePrefixMapper;
import xmldoc.sax.DocumentHandler;
import xmldoc.sax.DocumentHandlerImpl;
import xmldoc.sax.SAXExceptionDecoder;
import bizrules.registry.BusinessRuleRegistry;
import bizrules.registry.BusinessRuleRegistryImpl;

import com.gsl.sax.decoder.xerces.erss.XercesExceptionDecoder;

/**
 * This class encapsulates the customisable components that are used by the
 * document validator framework. There are default implementations for all
 * service components, so a standard validator implementation does not need to
 * be concerned with this class. However a validator may need to provide
 * alternative implementations for one or more components in order to handle
 * specific requirements. In this case, this class can be extended to create the
 * alternative component implementations in favour of the defaults.
 * <p>
 * All the methods on this class are protected. For each component, there is a
 * get...() method and a create...() method. The framework calls the get methods
 * to get handles on the components. The get methods should not be overridden.
 * Subclasses should override the create methods to create the components as
 * required. Overridden create methods may themselves call the get methods if
 * the components being created need to reference other components (watch out
 * for circular references though!).
 * <p>
 * The get methods may be called more than once. The create methods will only be
 * called once.
 * <p>
 * The only exception to this pattern are the two createInputSource(...) methods -
 * one creates an InputSource from a Reader, the other from an InputStream.
 * These methods do not have corresponding get...() methods;
 * <p>
 * 
 * @see DocumentValidator#createServices()
 * 
 * @author jesse
 */
public class DocumentValidatorServices {

    // the validation parameters

    private final ValidationParams params;

    // the components

    private DocumentHandler docHandler;

    protected BusinessRuleRegistry ruleRegistry;

    private ProcessingErrorCollector errorCollector;

    private SAXExceptionDecoder exceptionDecoder;

    private NamespacePrefixMapper namespacePrefixMapper;

    /**
     * Constructs the default services.
     */
    public DocumentValidatorServices(final ValidationParams params) {
        this.params = params;
    }

    /**
     * Protected access to params. The params instance itself is not made
     * protected in order to preserve the common convenience practice of
     * creating a validator specific services subclass as an inner class to the
     * validator.
     */
    protected ValidationParams params() {
        return params;
    }

    /**
     * This method is called by the framework to get the DocumentHandler to use
     * during validation.
     * <p>
     * This method is not designed to be overridden. Override
     * {@link DocumentValidatorServices#createDocumentHandler())} instead.
     * 
     * @return The DocumentHandler instance to be used during validation.
     */
    protected final DocumentHandler getDocumentHandler() {
        if (docHandler == null) {
            docHandler = createDocumentHandler();
        }
        return docHandler;
    }

    /**
     * This method is called by the framework to get the BusinessRuleRegistry to
     * use during validation.
     * <p>
     * This method is not designed to be overridden. Override
     * {@link DocumentValidatorServices#createRuleRegistry())} instead.
     * 
     * @return The BusinessRuleRegistry instance to be used during validation.
     */
    protected final BusinessRuleRegistry getRuleRegistry() {
        if (ruleRegistry == null) {
            ruleRegistry = createRuleRegistry();
        }
        return ruleRegistry;
    }

    /**
     * This method is called by the framework to get the
     * ProcessingErrorCollector to use during validation.
     * <p>
     * This method is not designed to be overridden. Override
     * {@link DocumentValidatorServices#createErrorCollector()} instead.
     * 
     * @return The ProcessingErrorCollector instance to be used during
     *         validation.
     */
    protected final ProcessingErrorCollector getErrorCollector() {
        if (errorCollector == null) {
            errorCollector = createErrorCollector();
        }
        return errorCollector;
    }

    /**
     * This method is called by the framework to get the SAXExceptionDecoder to
     * use during validation.
     * <p>
     * This method is not designed to be overridden. Override
     * {@link DocumentValidatorServices#createExceptionDecoder()} instead.
     * 
     * @return The SAXExceptionDecoder instance to be used during validation.
     */
    protected final SAXExceptionDecoder getExceptionDecoder() {
        if (exceptionDecoder == null) {
            exceptionDecoder = createExceptionDecoder();
        }
        return exceptionDecoder;
    }

    /**
     * This method is called by the framework to get the NamespacePrefixMapper
     * to use during validation.
     * <p>
     * This method is not designed to be overridden. Override
     * {@link DocumentValidatorServices#createNamespacePrefixMapper()} instead.
     * 
     * @return The NamespacePrefixMapper instance to be used during validation.
     */
    protected final NamespacePrefixMapper getNamespacePrefixMapper() {
        if (namespacePrefixMapper == null) {
            namespacePrefixMapper = createNamespacePrefixMapper();
        }
        return namespacePrefixMapper;
    }

    /**
     * This method is responsible for creating the DocumentHandler instance that
     * will be used during validation. By default this method creates an
     * instance of DocumentHandlerImpl.
     * <p>
     * Subclasses may override this method to use an alternative DocumentHandler
     * implementation.
     * 
     * @see DocumentHandlerImpl
     * 
     * @return The DocumentHandler instance to be used during validation.
     */
    protected DocumentHandler createDocumentHandler() {
        return new DocumentHandlerImpl(getRuleRegistry());
    }

    /**
     * This method is responsible for creating the BusinessRuleRegistry instance
     * that will be used during validation. By default this method creates an
     * instance of BusinessRuleRegistryImpl.
     * <p>
     * Subclasses may override this method to use an alternative
     * BusinessRuleRegistry implementation.
     * 
     * @see BusinessRuleRegistryImpl
     * 
     * @return The BusinessRuleRegistry instance to be used during validation.
     */
    protected BusinessRuleRegistry createRuleRegistry() {
        return new BusinessRuleRegistryImpl(getErrorCollector(), params.getBizruleParams());
    }

    /**
     * This method is responsible for creating the ProcessingErrorCollector
     * instance that will be used during validation. By default this method
     * creates an instance of SimpleErrorCollector.
     * <p>
     * Subclasses may override this method to use an alternative
     * ProcessingErrorCollector implementation.
     * 
     * @see SimpleErrorCollector
     * 
     * @return The ProcessingErrorCollector instance to be used during
     *         validation.
     */
    protected ProcessingErrorCollector createErrorCollector() {
        final SimpleErrorCollector collector = new SimpleErrorCollector();
        collector.setMaxErrors(params.getMaxErrors());
        return collector;
    }

    /**
     * This method is responsible for creating the SAXExceptionDecoder instance
     * that will be used during validation. By default this method creates an
     * instance of XercesExceptionDecoder.
     * <p>
     * Subclasses may override this method to use an alternative
     * SAXExceptionDecoder implementation.
     * 
     * @see XercesExceptionDecoder
     * 
     * @return The SAXExceptionDecoder instance to be used during validation.
     */
    protected SAXExceptionDecoder createExceptionDecoder() {
        return new XercesExceptionDecoder();
    }

    /**
     * This method is responsible for creating the NamespacePrefixMapper
     * instance that will be used during validation. By default this method
     * returns null;
     * <p>
     * Subclasses may override this method to use an alternative
     * NamespacePrefixMapper implementation.
     * 
     * @see NamespacePrefixMapper
     * 
     * @return The NamespacePrefixMapper instance to be used during validation.
     */
    protected NamespacePrefixMapper createNamespacePrefixMapper() {
        return null;
    }

    /**
     * This method is responsible for creating the InputSource instance that
     * will be provided to the SAX parser.
     * 
     * @param input
     *            The Reader instance where the input XML document can be read
     *            from.
     * 
     * @return The InputSource to provide to the SAX parser.
     * 
     * @throws IOException
     *             if there was an IO problem creating the input source.
     */
    protected InputSource createInputSource(final Reader input) throws IOException {
        return new InputSource(input);
    }

    /**
     * This method is responsible for creating the InputSource instance that
     * will be provided to the SAX parser.
     * 
     * @param input
     *            The InputStream from which the XML document can be read
     * 
     * @return The InputSource to provide to the SAX parser.
     * 
     * @throws IOException
     *             if there was an IO problem creating the input source.
     */
    protected InputSource createInputSource(final InputStream input) throws IOException {
        return new InputSource(input);
    }
}
