package com.gsl.docValidator;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import xmldoc.ParserParams;
import bizrules.params.BizrulesParams;

import com.gsl.logging.LoggingParams;

/**
 * Base class for providing parameters for validator implementations.
 * <p>
 * Provides convenience implementations of methods for getting and setting
 * arbitrary Object-typed attributes keyed by name.
 * <p>
 * This class should be extended by validator implementations wishing to cater
 * for document-type specific validation parameters.
 * 
 * @author jesse
 */
public abstract class ValidationParams implements Cloneable {

    private Map<Object, Object> parserFeatures = new HashMap<Object, Object>();

    private Map<String, Object> attributes = new HashMap<String, Object>();

    private LoggingParams loggingParams = new LoggingParams();

    private Date currentDate;

    private ParserParams parserParams = new ParserParams();

    private BizrulesParams bizruleParams = new BizrulesParams();

    private boolean includePrefixesInXPaths = true;

    private boolean outputDeveloperMessages = true;

    private boolean reportErrorLimitExceeded = true;

    private boolean suppressSchemaLocationHintErrors = false;

    private int maxErrors = 100000;

    @Override
    public Object clone() throws CloneNotSupportedException {
        final ValidationParams clone = (ValidationParams) super.clone();
        clone.loggingParams = (LoggingParams) loggingParams.clone();
        clone.parserParams = (ParserParams) parserParams.clone();
        clone.bizruleParams = (BizrulesParams) bizruleParams.clone();
        clone.parserFeatures = new HashMap<Object, Object>(parserFeatures);
        clone.attributes = new HashMap<String, Object>(attributes);

        return clone;
    }

    @Override
    public String toString() {
        String s = "ValidationParams[";

        s += "currentDate=" + (currentDate != null ? currentDate.toString() : "now");
        s += ", ";
        s += "includePrefixesInXPaths=" + includePrefixesInXPaths;
        s += ", ";
        s += "outputDeveloperMessages=" + outputDeveloperMessages;
        s += ", ";
        s += "reportErrorLimitExceeded=" + reportErrorLimitExceeded;
        s += ", ";
        s += "maxErrors=" + maxErrors;
        s += ", ";
        s += "attributes=" + attributes;
        s += ", ";
        s += "parserFeatures=" + parserFeatures;
        s += ",\n";
        s += "loggingParams=" + loggingParams;
        s += ",\n";
        s += "parserParams=" + parserParams;
        s += ",\n";
        s += "bizruleParams=" + bizruleParams;

        s += "]";
        return s;
    }

    /**
     * This method is used by the framework to provide the Xerces parser with a
     * grammar pool cache key. By default this method returns the fully
     * qualified name of the params implementation class. For simple validator
     * implementations this should be sufficient. Subclasses may override this
     * method to return a more suitable value if required.
     * 
     * @return An object to be used by the Xerces parser as a grammar pool cache
     *         key.
     */
    protected Object getVariant() {
        return getClass().getName();
    }

    /**
     * Returns a map of parser features that will be supplied to the Xerces
     * parser. May return null to mean no features.
     * 
     * @return A map of parser features to supply to the Xerces parser, or null.
     */
    public Map<Object, Object> getParserFeatures() {
        return parserFeatures;
    }

    /**
     * Add a feature that will be supplied to the Xerces parser.
     * 
     * @param name
     *            The name of the feature to add.
     * @param value
     *            The feature to add.
     */
    public void addParserFeature(final Object name, final Object value) {
        parserFeatures.put(name, value);
    }

    /**
     * Sets the map of parser features that will be supplied to the Xerces
     * parser. May be set to null to mean no features.
     * <p>
     * This will replace any features previously added via the
     * addParserFeature() method.
     * 
     * @see #addParserFeature(Object, Object)
     * 
     * @param parserFeatures
     *            The map of parser features to supply to the Xerces parser, or
     *            null.
     * 
     * @deprecated
     */
    @Deprecated
    public void setParserFeatures(final Map<?, ?> parserFeatures) {
        this.parserFeatures = new HashMap<Object, Object>(parserFeatures);
    }

    /**
     * Gets a named attribute associated with this parameters instance. Returns
     * null if the named attrbute does not exist.
     * 
     * @param name
     *            The name of the attribute to get.
     * 
     * @return The named attribute, or null.
     */
    public Object getAttribute(final String name) {
        return attributes.get(name);
    }

    /**
     * Sets an attribute to be associated with this parameters instance.
     * 
     * @param name
     *            The name of the attribute to set.
     * @param value
     *            The attribute to set.
     */
    public void setAttribute(final String name, final Object value) {
        attributes.put(name, value);
    }

    /**
     * Gets the logging parameters associated with this parameters instance.
     * 
     * @return The logging parameters associated with this parameters instance.
     */
    public LoggingParams getLoggingParams() {
        return loggingParams;
    }

    /**
     * Sets the logging parameters associated with this parameters instance.
     * <p>
     * Should not be null.
     * 
     * @param loggingParams
     *            The logging parameters to associate with this parameters
     *            instance.
     */
    public void setLoggingParams(final LoggingParams loggingParams) {
        this.loggingParams = loggingParams;
    }

    /**
     * Clients should call this if they want the input document to be validated
     * with respect to a given date/time. The default behaviour is to use the
     * current system time.
     * 
     * @param currentDate
     *            the Date instance the validation framework will use as the
     *            current date/time, or null to use the current system time.
     */
    public void setCurrentDate(final Date currentDate) {
        this.currentDate = currentDate;
    }

    /**
     * This method is package-private, it is only called by the validation
     * framework to set up the current date/time. Any code that needs to get the
     * current date should call {@link CurrentDate#getCurrentDate()}.
     * <p>
     * When called from the thread running the validation,
     * {@link CurrentDate#getCurrentDate()} will return the current date
     * depending on the value that was passed to {@link #setCurrentDate(Date)}.
     * 
     * @see #setCurrentDate(Date)
     * @see CurrentDate#getCurrentDate()
     * 
     * @return The current date/time as set by the client, or null if none was
     *         set.
     */
    public Date getCurrentDate() {
        return currentDate;
    }

    public BizrulesParams getBizruleParams() {
        return bizruleParams;
    }

    public void setBizruleParams(final BizrulesParams bizruleParams) {
        this.bizruleParams = bizruleParams;
    }

    public int getMaxErrors() {
        return maxErrors;
    }

    /**
     * This is the maximum number of errors that a document can generate before
     * the processing aborts. If set to 0 then there is no limit on the error
     * count.
     */
    public void setMaxErrors(final int maxErrors) {
        this.maxErrors = maxErrors;
    }

    public ParserParams getParserParams() {
        return parserParams;
    }

    public void setParserParams(final ParserParams parserParams) {
        this.parserParams = parserParams;
    }

    public boolean getIncludePrefixesInXPaths() {
        return includePrefixesInXPaths;
    }

    public void setIncludePrefixesInXPaths(final boolean includePrefixesInXPaths) {
        this.includePrefixesInXPaths = includePrefixesInXPaths;
    }

    public boolean getOutputDeveloperMessages() {
        return outputDeveloperMessages;
    }

    public void setOutputDeveloperMessages(final boolean outputDeveloperMessages) {
        this.outputDeveloperMessages = outputDeveloperMessages;
    }

    public boolean getReportErrorLimitExceeded() {
        return reportErrorLimitExceeded;
    }

    public void setReportErrorLimitExceeded(final boolean reportErrorLimitExceeded) {
        this.reportErrorLimitExceeded = reportErrorLimitExceeded;
    }

    /**
     * Setting this to true will prevent an error being reported if the
     * processor attempts to resolve a schemaLocation hint and fails.
     * 
     * Default is false
     * 
     * @param suppressSchemaLocationHintErrors
     */
    public void setSuppressSchemaLocationHintErrors(boolean suppressSchemaLocationHintErrors) {
        this.suppressSchemaLocationHintErrors = suppressSchemaLocationHintErrors;
    }

    public boolean getSuppressSchemaLocationHintErrors() {
        return suppressSchemaLocationHintErrors;
    }
}
