package com.gsl.docValidator;

import xmldoc.util.EntityResolver;

/**
 * This interface represents the logic required to handle one or more namespaces
 * that will be encountered during the validation of a given input document.
 * Multiple providers can be configured for a single validator implementation.
 * <p>
 * For instance, an input document might consist of an envelope, within which is
 * embedded the document to be validated. With the envelope and the document
 * itself in two different XML namespaces, individual ValidatorProvider
 * implementations can be created to deal with different parts of the whole
 * document. This allows both a seperation of concerns and an opportunity for
 * reusability. For example, a different document type might need to be
 * validated, but if it is wrapped in the same envelope, the chances are the
 * provider for the envelope can be re-used without much effort.
 * <p>
 * Another possibility is that a single validator implementation may be
 * configured to handle multiple input document types (in different namespaces).
 * When validating a given document, only the providers that pertain to the
 * document will be active, allowing the validation process to be configured at
 * runtime as needed (eg. to register a different set of rules depending on the
 * document type).
 * <p>
 * Note that getEntityResolver() is called prior to init().
 * 
 * @author jesse
 */
public interface ValidatorProvider {

    /**
     * Returns an EntityResolver instance configured to resolve entities for the
     * document(s) this provider intends to handle.
     * <p>
     * Note this method is called before init().
     * 
     * @param params
     *            The validation parameters for the current validation.
     * 
     * @return An EntityResolver instance configured appropriately.
     */
    EntityResolver getEntityResolver(ValidationParams params);

    /**
     * This gets called for all registered providers before the parse is
     * started.
     * <p>
     * ValidatorProvider implementations may initialise state, add business
     * rules to the validator, etc. as required.
     * <p>
     * Note this method is called after getEntityResolver().
     * 
     * @param validator
     *            The DocumentValidatorImplementor instance for the current
     *            validation.
     */
    void init(DocumentValidatorImplementor validator);

    /**
     * The framework invokes this method on all registered providers when a
     * namespace begins. The implementation of this method should first check
     * that the passed in namespace matches one that it is interested in
     * provisioning. If is not interested, this method should do nothing and
     * return immediately.
     * <p>
     * When this method is passed a namespace it is interested in, it should use
     * the passed in DocumentValidatorImplementor instance to configure the
     * validator as required. For instance, adding rules and element processors,
     * further configuring the post validation delegate, etc.
     * 
     * @param namespace
     *            The namespace from being provisioned.
     * @param validator
     *            The DocumentValidatorImplementor instance for the current
     *            validation.
     */
    void provisionNamespace(String namespace, DocumentValidatorImplementor validator);

    /**
     * This gets called after the parse is completed.
     */
    void destroy();
}
