/**
 * 
 */
package com.gsl.hmrc.paye.paper;

import static org.xmlpull.v1.XmlPullParser.END_DOCUMENT;
import static org.xmlpull.v1.XmlPullParser.END_TAG;
import static org.xmlpull.v1.XmlPullParser.START_TAG;

import java.io.IOException;
import java.io.Reader;
import java.util.HashMap;
import java.util.Map;

import org.xmlpull.mxp1.MXParser;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

import paye.eoy.bizrules.PAYEIRNamespaceMapping;
import paye.multiyear.TaxYear;

/**
 * @author doug
 * 
 */
public class FixupReader extends Reader {

    private static final String UID_BUFFER = "uidBuffer";

    private static final String PRE_NI_CS_BUFFER = "preNICsBuffer";

    private final XmlPullParser parser = new MXParser();

    private final LeftoverBuffer leftover = new LeftoverBuffer("<?xml version=\"1.0\"?>");

    private final Reader underlying;

    private final Map<String, String> buffers = new HashMap<String, String>();

    private final PAYEIRNamespaceMapping nsMapping = PAYEIRNamespaceMapping.getInstance();

    private final TaxYear taxYear;

    /**
     * During processing this can get switched to false if no further
     * transformation is required.
     */
    private boolean isActive = true;

    /**
     * Keep track of whether we are inside a P14 structure so we can decide if
     * we need to buffer certain parts of it. This is to avoid a clash with
     * similarly-named elements in a P35. See case 682 for more details.
     */
    private boolean inP14 = false;

    public FixupReader(final Reader input, final TaxYear taxYear) {
        this.underlying = input;
        this.taxYear = taxYear;
        try {
            parser.setFeature(MXParser.FEATURE_PROCESS_NAMESPACES, true);
            parser.setInput(input);
        } catch (final XmlPullParserException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void close() throws IOException {
        underlying.close();
    }

    @Override
    public int read(final char[] buf, final int offset, final int length) throws IOException {
        final int result;

        if (leftover.remaining() > 0) {
            final int readFromLeftover = leftover.read(buf, offset, length);

            // Any space left in the buffer?
            if (length - readFromLeftover > 0) {
                // Recursively call read() to pick up any more characters
                final int furtherReadCount = read(buf, offset + readFromLeftover, length - readFromLeftover);

                // Annoying check in case further read returned -1 for end of
                // input
                result = (furtherReadCount >= 0) ? readFromLeftover + furtherReadCount : readFromLeftover;
            } else {
                result = readFromLeftover;
            }
        } else {
            result = readFromTokenStream(buf, offset, length);
        }
        return result;
    }

    private int readFromTokenStream(final char[] buf, final int offset, final int length) throws IOException {
        final int result;
        int adjustedOffset = offset;
        int adjustedLength = length;
        int totalRead = 0;
        while (adjustedLength > 0) {
            final int readCount = readNextToken(buf, adjustedOffset, adjustedLength);
            if (readCount == -1) {
                break;
            } else {
                totalRead += readCount;
                adjustedOffset += readCount;
                adjustedLength -= readCount;
            }
        }
        result = totalRead > 0 ? totalRead : -1;
        return result;
    }

    /**
     * Read the next token from the underlying XML stream and act on it. Fill
     * the incoming buffer with any characters that are ready for streaming on
     * to the caller.
     * 
     * @return the number of characters copied into the callers buffer.
     * @throws IOException
     */
    private int readNextToken(final char[] buf, final int offset, final int length) throws IOException {
        int token;
        try {
            /*
             * Because of the leftover buffering, we can get called twice at the
             * end of the document. Check if we're already there, otherwise the
             * call to nextToken() will throw an exception.
             */
            if (parser.getEventType() == END_DOCUMENT) {
                return -1;
            } else {
                token = parser.nextToken();
            }
        } catch (final XmlPullParserException e) {
            throw new RuntimeException(e);
        }

        final int[] startLength = new int[2];
        if (token != END_DOCUMENT) {
            char[] chars;
            int tokenStart;
            int tokenLength;

            final String modified = dispatchToken(token);
            if (modified != null) {
                chars = modified.toCharArray();
                tokenStart = 0;
                tokenLength = chars.length;
            } else {
                chars = parser.getTextCharacters(startLength);
                tokenLength = startLength[1];
                tokenStart = startLength[0];
            }
            final int numberToCopy = (tokenLength < length) ? tokenLength : length;
            System.arraycopy(chars, tokenStart, buf, offset, numberToCopy);

            if (numberToCopy < tokenLength) {
                final int leftoverCount = tokenLength - numberToCopy;
                leftover.deposit(chars, tokenStart + (tokenLength - leftoverCount), leftoverCount);
            }

            return numberToCopy;
        } else {
            return -1;
        }
    }

    /**
     * 
     * @param token
     * @return the transformed text from the dispatch, or null if the text was
     *         not transformed.
     * @throws IOException
     */
    private String dispatchToken(final int token) throws IOException {
        if (!isActive) {
            // Transformation has been disabled.
            return null;
        }

        try {
            if (token == START_TAG) {
                final String elementName = parser.getName();
                if (elementName.equals("ArchiveNumber")) {
                    return removeElement();
                } else if (elementName.equals("P14") || elementName.equals("P14Amended")) {
                    inP14 = true;
                } else if (inP14 && elementName.equals("SSP")) {
                    return bufferUntilEndOf("PayAndTax", PRE_NI_CS_BUFFER);
                } else if (elementName.equals("UniqueID")) {
                    return bufferUntilEndOf("UniqueID", UID_BUFFER);
                } else if (elementName.equals("IRenvelope")) {
                    final String namespace = parser.getNamespace();
                    if (!namespace.startsWith("http://www.govtalk.gov.uk/taxation/EOY-consolidated")) {
                        return "<IRenvelope xmlns=\"" + nsMapping.getIRNamespaceForTaxYear(taxYear) + "\">";
                    } else {
                        /*
                         * The IRenvelope namespace was for a consolidated
                         * submission so switch off any further re-writing.
                         */
                        isActive = false;
                        return null;
                    }
                } else if (elementName.equals("EricEnvelope")) {
                    return "<EricEnvelope xmlns=\"http://www.inlandrevenue.gov.uk/schemas/EricEnvelope\">";
                }
            } else if (token == END_TAG) {
                final String elementName = parser.getName();
                if (elementName.equals("P14") || elementName.equals("P14Amended")) {
                    inP14 = false;
                    return getBufferText(PRE_NI_CS_BUFFER) + parser.getText();
                } else if (elementName.equals("EndOfYearReturn")) {
                    return getBufferText(UID_BUFFER) + parser.getText();
                }
            }
        } catch (final XmlPullParserException e) {
            throw new RuntimeException(e);
        }
        return null;
    }

    private String getBufferText(final String bufferName) {
        String bufferText = buffers.get(bufferName);
        if (bufferText == null) {
            bufferText = "";
        }
        return bufferText;
    }

    private String bufferUntilEndOf(final String endElementName, final String bufferName) throws XmlPullParserException,
            IOException {
        final StringBuffer buf = new StringBuffer(parser.getText());
        while (true) {
            final int token = parser.nextToken();
            final String text = parser.getText();

            if (token == START_TAG && parser.isEmptyElementTag()) {
                /*
                 * Both the start and end tag for an empty element like
                 * <"Code"/> report the same text, so skip it on the start tag
                 * and we'll pick it up on the end tag
                 */
            } else {
                buf.append(text);
            }

            final String name = parser.getName();
            if (token == END_TAG && name.equals(endElementName)) {
                break;
            }
        }
        buffers.put(bufferName, buf.toString());
        return "";
    }

    private String removeElement() throws IOException, XmlPullParserException {
        while (parser.next() != END_TAG) {
            // do nothing, swallow the tag and its contents
        }
        return "";
    }
}
