/**
 * 
 */
package com.gsl.hmrc.paye.paper;

import java.io.IOException;

/**
 * In a situation where you are filtering an underlying stream, you sometimes
 * read more characters from the underlying stream than you can return to the
 * caller because the buffer the caller provided is not big enough to hold them
 * all. In that case, you can deposit them into a LeftoverBuffer so that you can
 * pull them in later.
 * 
 * @author douglas
 * 
 */
public class LeftoverBuffer {

    private char[] leftover = null;

    private int position = -1;

    private int length = 0;

    public LeftoverBuffer() {
        // do nothing
    }

    /**
     * Create a LeftoverReader populated with the initial value
     * 
     * @param initialValue
     */
    public LeftoverBuffer(final String initialValue) {
        leftover = initialValue.toCharArray();
        position = 0;
        length = leftover.length;
    }

    /**
     * Read characters out of the leftover buffer.
     * 
     * @return 0 if there are no characters in the leftover buffer, otherwise
     *         the number of characters copied into the supplied buffer.
     */
    public int read(final char[] cbuf, final int off, final int len) throws IOException {
        if (remaining() > 0) {
            if (remaining() > len) {
                // We have more than enough chars to fill the requested buffer
                System.arraycopy(leftover, position, cbuf, off, len);
                position += len;
                length -= len;
                return len;
            } else {
                // Filling the buffer will use up all of our chars
                System.arraycopy(leftover, position, cbuf, off, length);
                final int numCopied = length;
                leftover = null;
                position = -1;
                length = 0;
                return numCopied;
            }
        } else {
            return 0;
        }
    }

    public void deposit(final char[] cbuf, final int off, final int len) {
        if (leftover == null) {
            leftover = new char[len];
            System.arraycopy(cbuf, off, leftover, 0, len);
            position = 0;
            length = leftover.length;
        } else {
            final char[] newLeftover = new char[length + len];
            System.arraycopy(leftover, position, newLeftover, 0, length);
            System.arraycopy(cbuf, off, newLeftover, length, len);
            leftover = newLeftover;
            position = 0;
            length = leftover.length;
        }
    }

    /**
     * @return the number of characters remaining to be read from the character
     *         buffer, or 0 if there are none left.
     */
    public int remaining() {
        return length;
    }
}
