/*
 * Created on Jul 20, 2005
 * 
 * $Id: FileUtils.java,v 1.3 2005/10/03 09:56:52 dec Exp $
 */
package com.gsl.io.core.file;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.util.regex.Pattern;

import com.gsl.io.core.stream.StreamUtils;

/**
 * @author Jaco
 * @since Jul 20, 2005
 */
public class FileUtils {

    private static final Pattern nonZeroLengthStringPattern = Pattern.compile(".+");

    /**
     * The file named by <em>filename</em> will be created. Any directories on
     * the path in the filename which do not already exist will be created as
     * well.
     * 
     * @since Oct 3, 2005
     */
    public static File createFileInstance(final String filename) {
        File ret = new File(filename);
        if (!ret.exists()) {
            createParentDirectory(filename);
            ret = new File(filename);
        }
        return ret;
    }

    /**
     * Ensure that all the parent directories of the filename exist, creating
     * them if necessary.
     */
    public static void createParentDirectory(final String filename) {
        final String[] dirs = filename.split(File.separator.equals("\\") ? "\\" + File.separator : File.separator);
        String dirName = "";
        for (int i = 0; i < dirs.length - 1; i++) {
            dirName += dirs[i] + File.separator;
            if (!new File(dirName).exists()) {
                new File(dirName).mkdir();
            }
        }
    }

    /**
     * The files named in the array <em>filesToStitch</em> will be
     * concatenated and written to <em>finalName</em>.
     * 
     * @since Oct 3, 2005
     */
    public static void stitch(final String[] filesToStitch, final String finalName) throws IOException {
        final FileWriter w = new FileWriter(finalName);

        for (final String element : filesToStitch) {
            final FileReader reader = new FileReader(element);
            final char[] buffer = new char[1024];
            int charsRead = 0;
            while ((charsRead = reader.read(buffer)) > 0) {
                w.write(buffer, 0, charsRead);
            }
            reader.close();
        }
        w.close();
    }

    /**
     * All the files named in the array <em>filesToZap</em> will be deleted.
     * 
     * @since Oct 3, 2005
     */
    public static void zap(final String[] filesToZap) {
        for (final String element : filesToZap) {
            final File f = new File(element);
            f.delete();
        }
    }

    public static boolean exists(final String fileName) {
        final File file = new File(fileName);
        return file.exists();
    }

    /**
     * Copy a file. The from argument must be a file. The to argument may be a
     * file or a directory.
     */
    public static void copy(final File from, File to) throws IOException {
        if (to.isDirectory()) {
            to = new File(to, from.getName());
        }
        final FileInputStream fis = new FileInputStream(from);
        final FileOutputStream fos = new FileOutputStream(to);

        try {
            final byte[] bytes = StreamUtils.readAllBytesFromStream(fis);
            fos.write(bytes);
        } finally {
            fos.close();
        }
    }

    /**
     * Delete all files and empty subdirectories in a directory. This is not a
     * recursive delete i.e. any files in subdirectories are not deleted.
     * 
     * @param dir
     *            the directory where files are to be deleted.
     */
    public static void deleteAllFilesInDir(final File dir) {
        deleteFilesInDir(dir, nonZeroLengthStringPattern);
    }

    /**
     * Delete files and empty subdirectories, whose name matches a regex
     * pattern, in a directory. This is not a recursive delete i.e. any files in
     * subdirectories are not deleted.
     * 
     * @param dir
     *            the directory where files are to be deleted.
     * @param filenamePattern
     *            the regex pattern used to match the names of files to be
     *            deleted.
     */
    public static void deleteFilesInDir(final File dir, final String filenamePattern) {
        deleteFilesInDir(dir, Pattern.compile(filenamePattern));
    }

    /**
     * Delete files and empty subdirectories, whose name matches a regex
     * pattern, in a directory. This is not a recursive delete i.e. any files in
     * subdirectories are not deleted.
     * 
     * @param dir
     *            the directory where files are to be deleted.
     * @param filenamePattern
     *            the regex pattern used to match the names of files to be
     *            deleted.
     */
    public static void deleteFilesInDir(final File dir, final Pattern filenamePattern) {
        if (dir.exists() && dir.isDirectory()) {
            final File[] matchingFiles = dir.listFiles(new RegexFilenameFilter(filenamePattern));

            for (final File element : matchingFiles) {
                element.delete();
            }
        }
    }

    /**
     * Recursively delete all files in a directory and its subdirectories.
     * Careful!
     * 
     * @param dir
     *            the top directory where files are to be deleted.
     */
    public static void deleteRecursivelyAllFilesInDir(final File dir) {
        deleteRecursivelyFilesInDir(dir, nonZeroLengthStringPattern);
    }

    /**
     * Recursively delete files, whose name matches a regex pattern, in a
     * directory and its subdirectories,
     * 
     * @param dir
     *            the top directory where files are to be deleted.
     * @param filenamePattern
     *            the regex pattern used to match the names of files to be
     *            deleted.
     */
    public static void deleteRecursivelyFilesInDir(final File dir, final String filenamePattern) {
        deleteRecursivelyFilesInDir(dir, Pattern.compile(filenamePattern));
    }

    /**
     * Recursively delete files, whose name matches a regex pattern, in a
     * directory and its subdirectories.
     * 
     * @param dir
     *            the top directory where files are to be deleted.
     * @param filenamePattern
     *            the regex pattern used to match the names of files to be
     *            deleted.
     */
    public static void deleteRecursivelyFilesInDir(final File dir, final Pattern filenamePattern) {
        deleteFilesInDir(dir, filenamePattern);

        if (dir.exists() && dir.isDirectory()) {
            final File[] children = dir.listFiles();

            for (final File element : children) {
                deleteRecursivelyFilesInDir(element, filenamePattern);
            }
        }
    }

    public static void clearDirectory(final String directoryName) {
        final File directory = new File(directoryName);
        clearDirectory(directory);
    }

    public static void clearDirectory(final File file) {
        if (!file.exists()) {
            file.mkdirs();
        } else {
            if (!file.isDirectory()) {
                System.out.println("Cannot clear directory " + file);
                System.exit(1);
            } else {
                final File[] files = file.listFiles();
                for (final File f : files) {
                    if (f.isDirectory()) {
                        clearDirectory(f);
                    }
                    f.delete();
                }
            }
        }
    }

    public static byte[] readFileToBytes(final File inputFile) throws FileNotFoundException, IOException {
        return readFileToString(inputFile).getBytes();
    }

    public static byte[] readSteamToBytes(final InputStream is) throws IOException {
        return readStreamToString(is).getBytes();
    }

    public static String readFileToString(final File inputFile) throws FileNotFoundException, IOException {
        final FileInputStream fis = new FileInputStream(inputFile);

        return readStreamToString(fis);
    }

    public static String readStreamToString(final InputStream fis) throws IOException {
        final byte[] b = new byte[1024];

        String input = "";

        int count = -1;
        while ((count = fis.read(b)) >= 0) {
            input += new String(b, 0, count);
        }
        return input;
    }
}
