package com.gsl.io.core.file;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;

import com.gsl.io.core.stream.StreamUtils;

/**
 * Builds a JAR file.
 * 
 * @author rhys
 */

public class JarBuilder {

    /**
     * Builds a JAR file with the specified filename at the specified path. The
     * entries root directory is searched recursively for all directories and
     * files; for each one found, an entry is added to the JAR file.
     * 
     * @param entriesRootDirpath
     *            the path to the top directory containing entries to be added
     *            to the JAR file.
     * @param jarFilepath
     *            the filepath (directory path plus filename) of the JAR file to
     *            be built. The directory path will be created if it does not
     *            exist.
     * @throws IOException
     *             if the JAR file cannot be written due to a file I/O error.
     */
    public static void buildJar(String entriesRootDirpath, final String jarFilepath) throws IOException {
        final File jarFile = new File(jarFilepath);
        if (jarFile.isDirectory()) {
            throw new IOException("JAR path \"" + jarFilepath + "\" points to a directory. It must denote a path to a file.");
        }

        if (!entriesRootDirpath.endsWith("/")) {
            entriesRootDirpath += "/";
        }

        // Collect all file entries to be placed in JAR
        final File entryRootDir = new File(entriesRootDirpath);
        final Set<File> entryFiles = new LinkedHashSet<File>();
        collectJarEntries(entryRootDir, entryFiles);

        // Make directory path for JAR file if it doesn't exist
        final File jarFileDir = jarFile.getParentFile();
        if ((jarFileDir != null) && !jarFileDir.exists()) {
            jarFileDir.mkdirs();
        }

        // Write JAR
        // Buffered output stream isn't required, in fact it causes trouble with
        // entries in JARs.
        final Manifest manifest = new Manifest();
        final JarOutputStream jos = new JarOutputStream(new FileOutputStream(jarFile), manifest);

        try {
            for (final File entryFile : entryFiles) {

                final String entryFilepath = makeEntryFilepath(entriesRootDirpath, entryFile);

                final JarEntry entry = new JarEntry(entryFilepath);
                jos.putNextEntry(entry);

                if (entryFile.isFile()) {
                    final InputStream is = new FileInputStream(entryFile);

                    try {
                        final byte[] classFileBytes = StreamUtils.readAllBytesFromStream(is);
                        jos.write(classFileBytes);
                    } finally {
                        is.close();
                    }
                }
            }
        } finally {
            jos.close();
        }
    }

    private static String makeEntryFilepath(final String entriesRootDirpath, final File entryFile) {
        final StringBuffer buffer = new StringBuffer();

        final String entryFilepath = entryFile.getAbsolutePath();
        if (entryFilepath.startsWith(entriesRootDirpath)) {
            buffer.append(entryFilepath.substring(entriesRootDirpath.length()));
        } else {
            buffer.append(entryFilepath);
        }

        if (entryFile.isDirectory() && !entryFilepath.endsWith(File.separator)) {
            buffer.append(File.separator);
        }

        return buffer.toString();
    }

    private static void collectJarEntries(final File dir, final Set<File> entryFiles) {
        if (dir.exists() && dir.isDirectory() && dir.canRead()) {
            final File[] filesFound = dir.listFiles();

            for (final File entryFile : filesFound) {
                entryFiles.add(entryFile);
            }

            // Traverse child directories recursively
            final File[] children = dir.listFiles();

            for (final File element : children) {
                collectJarEntries(element, entryFiles);
            }
        }
    }
}
