/*
 * Created on Oct 9, 2005
 *
 * $Id: RecursiveFileProvider.java,v 1.2 2005/10/11 10:03:03 dec Exp $
 */
package com.gsl.io.core.file;

import java.io.File;
import java.io.FilenameFilter;
import java.util.Iterator;
import java.util.Set;
import java.util.logging.Logger;

/**
 * This implementation takes a base directory and searches with that directory
 * and its subdirectories for files. You can optionally provide a FilenameFilter
 * to restrict the files which are selected.
 * 
 * @author douglas
 * @since Oct 9, 2005
 */
public class RecursiveFileProvider implements FileProvider {
    private final Set<File> selectedFiles;

    private final Iterator<File> selectedFileIterator;

    private static Logger logger = Logger.getLogger("io.core.file.RecursiveFileProvider");

    /**
     * Number of Files already provided.
     */
    private long providedCount = 0;

    public RecursiveFileProvider(final String baseDirectoryName) {
        this(baseDirectoryName, new AcceptAllFilenameFilter());
    }

    public RecursiveFileProvider(final String baseDirectoryName, final FilenameFilter filter) {
        final File baseDirectory = new File(baseDirectoryName);

        if (!baseDirectory.isDirectory()) {
            throw new IllegalArgumentException(baseDirectoryName + " is not a directory");
        }

        final FileSelector selector = new RecursiveFileSelector(filter);
        selectedFiles = selector.selectFiles(baseDirectory);
        selectedFileIterator = selectedFiles.iterator();
    }

    public synchronized File getNextFile() {
        File result = null;

        if (selectedFileIterator.hasNext()) {
            providedCount++;
            result = selectedFileIterator.next();
        } else {
            if (providedCount != getFileCount()) {
                logger.warning("Total files provided: " + providedCount + ", should have been " + getFileCount());
            }
        }

        return result;
    }

    public long getFileCount() {
        return selectedFiles.size();
    }

    /**
     * @return Returns the number of files already provided by this provider.
     */
    public synchronized long getProvidedCount() {
        return providedCount;
    }
}
