package com.gsl.io.core.file;

import java.io.File;
import java.util.Collection;

public class SuffixFileFilter extends AbstractIOFileFilter {

    private final String suffixes[];

    private final boolean caseInsensitive;

    public SuffixFileFilter(final String suffix) {
        this(suffix, true);
    }

    public SuffixFileFilter(final String suffix, final boolean caseInsensitive) {
        if (suffix == null) {
            throw new IllegalArgumentException("The suffix must not be null");
        }
        this.suffixes = (new String[] { suffix });
        this.caseInsensitive = caseInsensitive;
        if (caseInsensitive)
            makeSuffixesLowerCase();
    }

    public SuffixFileFilter(final String suffixes[]) {
        this(suffixes, true);
    }

    public SuffixFileFilter(final String suffixes[], final boolean caseInsensitive) {
        if (suffixes == null) {
            throw new IllegalArgumentException("The array of suffixes must not be null");
        }
        this.suffixes = suffixes;
        this.caseInsensitive = caseInsensitive;
        if (caseInsensitive)
            makeSuffixesLowerCase();
    }

    public SuffixFileFilter(final Collection<String> suffixes) {
        this(suffixes, true);
    }

    public SuffixFileFilter(final Collection<String> suffixes, final boolean caseInsensitive) {
        if (suffixes == null) {
            throw new IllegalArgumentException("The collection of suffixes must not be null");
        }
        this.suffixes = suffixes.toArray(new String[suffixes.size()]);
        this.caseInsensitive = caseInsensitive;
        if (caseInsensitive)
            makeSuffixesLowerCase();
    }

    private void makeSuffixesLowerCase() {
        for (int i = 0; i < suffixes.length; i++)
            suffixes[i] = suffixes[i].toLowerCase();
    }

    @Override
    public boolean accept(final File file) {
        String name = file.getName();
        if (caseInsensitive)
            name = name.toLowerCase();
        for (final String element : suffixes)
            if (name.endsWith(element))
                return true;
        return false;
    }

    @Override
    public boolean accept(final File file, String name) {
        if (caseInsensitive)
            name = name.toLowerCase();
        for (final String element : suffixes)
            if (name.endsWith(element))
                return true;
        return false;
    }
}
