/**
 * $Id: ProgressReader.java,v 1.4 2006/05/20 21:31:50 dec Exp $
 */
package com.gsl.io.core.reader;

import java.io.IOException;
import java.io.Reader;

/**
 * This is a convenience class which wraps an underlying Reader object. The
 * intent is that we will keep track of how far we have read through the
 * underlying data and make callbacks to a ProgressReciever object (provided by
 * the user at construction time) to indicate the percentage progress.
 * 
 * Since it is not possible to tell the length of the data on a generic Reader,
 * the user must tell us how long he expects the data stream to be at
 * construction time.
 * 
 * @author Douglas Clinton
 * @since Jan 18, 2006
 * 
 */
public class ProgressReader extends Reader {

    private final Reader underlyingReader;

    private long position = 0;

    private final long length;

    private final ProgressReceiver callback;

    /**
     * Create a new progress reader. Since a generic Reader does not provide a
     * way of determining the length of the input data (after all, the Reader
     * may be streaming and thus have no way of knowing), it is up to the user
     * of this class to say how long the data stream is. Of course, in some
     * cases this may need to be a guess and it is up to the user to decide how
     * best to calculate the length.
     * 
     * Every time a read() is performed on this Reader, we will call the
     * setProgress() method on the callback object to say what percentage of the
     * way through the data we have read.
     * 
     * @param underlyingReader
     *            the wrapped Reader. The actual data will be read from this
     * @param length
     *            the total number of characters of data which we expect to read
     *            from the wrapped Reader. It is up to the user to determine
     *            this. In cases where the underlying data is streamed, the user
     *            may have to provide an 'intelligent guess' for this.
     * @param callback
     *            the object which will receive the setProgress() callbacks as
     *            we read our way through the underlying data.
     */
    public ProgressReader(final Reader underlyingReader, final long length, final ProgressReceiver callback) {
        super();
        this.underlyingReader = underlyingReader;
        this.length = length;
        this.callback = callback;
    }

    /**
     * We will read a chunk of data from the underlying stream into the
     * character buffer and return the number of characters read (as per the
     * Reader interface spec), but
     */
    @Override
    public int read(final char[] buf, final int off, final int len) throws IOException {
        final int result = underlyingReader.read(buf, off, len);
        if (result != -1) {
            position += result;
        }

        callback.setProgress(getProgress());
        return result;
    }

    public int getProgress() {
        int result = 100;
        if (length > 0) {
            result = (int) (((position % length) * 100) / length);
        }
        return result;
    }

    @Override
    public void close() throws IOException {
        underlyingReader.close();
    }
}
