package com.gsl.io.core.stream;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * Stream-handling utility methods.
 * 
 * @author rhys
 */

public class StreamUtils {

    /**
     * Returns a byte array containing all bytes read from an input stream until
     * end-of-stream is reached.
     * <p>
     * 
     * The input stream passed in is always close()'d on return from this
     * method, so the caller should never expect to use the input stream after
     * this method returns.
     * 
     * @param is
     *            the input stream for which the byte array is sought.
     * @return a byte array containing all bytes read from the stream.
     * @throws IOException
     */
    public static byte[] readAllBytesFromStream(final InputStream is) throws IOException {
        final ByteArrayOutputStream baos = new ByteArrayOutputStream();

        /*
         * Common trap avoided: is.available() is never a reliable estimator of
         * the number of bytes remaining to be read before end-of-stream is
         * reached!
         */
        byte[] bytes = new byte[is.available() + 1];
        int bytesReadCount = is.read(bytes);

        try {
            while (bytesReadCount != -1) {
                baos.write(bytes, 0, bytesReadCount);

                bytes = new byte[is.available() + 1];
                bytesReadCount = is.read(bytes);
            }
        } finally {
            is.close();
            baos.close();
        }

        final byte[] allBytes = baos.toByteArray();
        return allBytes;
    }

    /**
     * Creates byte-equivalent copies of a stream. A byte-equivalent copy of
     * stream A is defined as a stream B, such that the full list of bytes read
     * from A using A.read() is identical to the full list of bytes read from B
     * using B.read() until A's end-of-stream is reached. This method makes no
     * guarantee of similarity between A and B other than byte equivalency; the
     * InputStream implementation of B may bear no resemblance whatsoever to A.
     * <p>
     * 
     * The input stream passed in is always close()'d on return from this
     * method, so the caller should never expect to use the input stream after
     * this method returns.
     * 
     * @param is
     *            the input stream of which byte-equivalent copies are sought.
     * @param copyCount
     *            the number of byte-equivalent stream copies to make.
     * @return an array containing the byte-equivalent stream copies.
     * @throws IOException
     */
    public static InputStream[] createByteEquivalentStreamCopies(final InputStream is, final int copyCount) throws IOException {
        final byte[] allBytes = readAllBytesFromStream(is);
        final InputStream[] streamCopies = new InputStream[copyCount];

        for (int i = 0; i < copyCount; i++) {
            final InputStream streamCopy = new ByteArrayInputStream(allBytes);
            streamCopies[i] = streamCopy;
        }

        return streamCopies;
    }

}
