package com.gsl.logging;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * This provides a thin wrapper around java.util.Logging. The main aim is to
 * allow log levels to be set by providing a Map which set levels for specific
 * logger names without intruding on the logging behaviour of a system in which
 * we might be embedded.
 * 
 * A 'scope' string can be provided which will be prepended to logger names to
 * avoid name clashes with the surrounding system.
 * 
 * @author Douglas Clinton
 * 
 */
public class LoggerFactory {
    private static String scope = "default";

    private static Level defaultDefaultLevel = Level.INFO;

    private static Map<String, Level> levels = new HashMap<String, Level>();

    private LoggerFactory() {
        // do nothing - prevent instantiation
    }

    /**
     * Creates or retrieves a java.util.Logger with the given name. The Logger's
     * level will be set from the 'levels' map.
     * 
     */
    public static Logger getLogger(final String name) {
        final Logger result = Logger.getLogger(getScopedName(name));
        result.setLevel(lookupLevel(name));

        return result;
    }

    private static String getScopedName(final String name) {
        return getScope() + "." + name;
    }

    /**
     * Find the appropriate log level for a logger by searching up the hierarchy
     * of its name. E.g. a name of 'x.y.z' would be matched first against
     * 'x.y.z', then 'x.y' and then 'x' in the level map, stopping when the
     * first match is found.
     * 
     * If no match is found then the level for an entry of 'default' will be
     * used and, failing that, the log level for an anonymous logger will be
     * used.
     * 
     * @param name
     * @return
     */
    private static Level lookupLevel(final String name) {
        Level result = null;

        String lookupString = name;

        boolean found = false;
        while (!found && !lookupString.equals("")) {
            final Level level = levels.get(lookupString);
            if (level != null) {
                result = level;
                found = true;
            } else {
                final int dot = lookupString.lastIndexOf('.');
                if (dot > 0) {
                    lookupString = lookupString.substring(0, dot);
                } else {
                    lookupString = "";
                }
            }
        }

        if (!found) {
            result = getDefaultLevel();
        }

        return result;
    }

    private static Level getDefaultLevel() {
        Level result = levels.get("default");

        if (result == null) {
            result = defaultDefaultLevel;
            levels.put("default", result);
        }

        return result;
    }

    public static String getScope() {
        return scope;
    }

    /**
     * Loggers will have this string prepended to their names (with a separating
     * '.') when they are created. This is to avoid naming clashes with any
     * other systems which may be using Loggers outside of this mechanism.
     * 
     */
    public static void setScope(final String scope) {
        LoggerFactory.scope = scope;
    }

    public static Map<String, Level> getLevels() {
        return levels;
    }

    /**
     * Provide a Map which will let us match logger names to levels. The naming
     * convention is hierarchical so a logger name of 'x.y.z' could match
     * entries in the map for 'x', 'x.y' or 'x.y.z', whichever is the most
     * speficic.
     * 
     * If no map entry matches an entry for 'default' will be used and if that
     * does not exist then the defaul log level for an anonymous logger will be
     * used.
     * 
     * Note that the names in the map should *not* include the 'scope' string at
     * the beginning. The 'scope' string is simply used internally to avoid name
     * clashes with a surrounding system. In fact, the whole 'scope' idea may be
     * useless and might be better removed or replaced.
     * 
     * @param levels
     */
    public static void setLevels(final Map<String, Level> levels) {
        LoggerFactory.levels = levels;
    }
}
