/**
 * 
 */
package com.gsl.reader.nscorrect;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.PushbackReader;
import java.io.Reader;

import paye.eoy.bizrules.PAYEIRNamespaceMapping;
import paye.multiyear.TaxYear;

/**
 * The EBS system that converts EDI submissions to XML does not version the PAYE
 * namespace year-on-year. It uses http://www.govtalk.gov.uk/taxation/EOY for
 * all submissions which means they do not get validated properly. This Reader
 * filters the incoming submission text looking for the unversioned namespace
 * and replacing instances of it with the versioned namespace for the tax year
 * of the submission.
 * <p>
 * See {@link https://gsl.fogbugz.com/default.asp?651}
 * 
 * @author doug
 * 
 */
public class NamespaceCorrectingReader extends Reader {

    private final PushbackReader underlying;

    private final String correctNamespace;

    private final String unversionedNamespace;

    private final boolean active;

    private final PAYEIRNamespaceMapping nsMapping = PAYEIRNamespaceMapping.getInstance();

    public NamespaceCorrectingReader(final Reader underlying, final TaxYear taxYear) {

        this.correctNamespace = "\"" + nsMapping.getIRNamespaceForTaxYear(taxYear) + "\"";
        this.unversionedNamespace = "\"" + nsMapping.getIRNamespaceForTaxYear(TaxYear.TAX_YEAR_0405) + "\"";

        final int pushbackCount = correctNamespace.length() > unversionedNamespace.length() ? correctNamespace.length()
                : unversionedNamespace.length();
        this.underlying = new PushbackReader(new BufferedReader(underlying), pushbackCount);

        /*
         * 0405 and 0506 use the unversioned namespace so turn off the filtering
         * for them.
         */
        active = !correctNamespace.equals(unversionedNamespace);
    }

    @Override
    public void close() throws IOException {
        underlying.close();
    }

    @Override
    public int read(final char[] buf, final int offset, final int length) throws IOException {
        if (!active) {
            return underlying.read(buf, offset, length);
        } else {
            boolean matching = false;
            final StringBuilder beforeMatchingText = new StringBuilder();
            StringBuilder matchingText = new StringBuilder();

            int pos = 0;
            StringBuilder result = null;
            int c = underlying.read();
            if (c == -1) {
                return -1;
            }
            do {
                if (c == unversionedNamespace.charAt(pos)) {
                    matching = true;
                    matchingText.append((char) c);
                    if (++pos == unversionedNamespace.length()) {
                        beforeMatchingText.append(correctNamespace);
                        result = beforeMatchingText;
                        break;
                    }
                } else {
                    if (matching) {
                        underlying.unread(c);
                        beforeMatchingText.append(matchingText);
                        matchingText = new StringBuilder();
                        matching = false;
                        pos = 0;
                    } else {
                        beforeMatchingText.append((char) c);
                    }
                    result = beforeMatchingText;
                    if (beforeMatchingText.length() >= length) {
                        break;
                    }
                }
            } while ((c = underlying.read()) >= 0);

            final int charsRead = (result.length() > length) ? length : result.length();
            for (int i = 0; i < charsRead; i++) {
                buf[offset + i] = result.charAt(i);
            }

            /*
             * We've ended up reading more characters from the underlying Reader
             * than our caller wanted so push the remainder back onto the
             * PushbackReader.
             */
            if (result.length() > length) {
                final char[] toUnread = new char[result.length() - charsRead];
                result.getChars(charsRead, result.length(), toUnread, 0);
                underlying.unread(toUnread);
            }
            return charsRead;
        }
    }
}
