/**
 * $Id: CVCPatternValidDecoder.java,v 1.2 2006/08/10 15:19:10 dec Exp $
 */
package com.gsl.sax.decoder.xerces.chris;

import org.xml.sax.SAXParseException;

import xmldoc.DocumentError;
import xmldoc.ElementReference;
import xmldoc.sax.DocumentErrorPostProcessor.FixupType;

import com.gsl.sax.decoder.xerces.AbstractXercesErrorDecoder;

/**
 * 
 * The order in which Xerces reports a cvc-pattern-valid error versus the start
 * element event for the field in error depends on how the XML is formatted.
 * When it looks like <foo></foo> (call it situation A for reference) then the
 * start element is reported before the error, so the location stack will be
 * correct and no fixup is needed.
 * 
 * However, if the xml looks like <foo/> (situation B) then the error is
 * reported before the start element event and the location stack still points
 * to the parent element at the time the error is raised.
 * 
 * Since there is no way for us to know which situation applies, this decoder
 * works on the basis that in situation A, the error event will be followed by
 * an end element event. In theory, there could be a sub-element of the element
 * in error, but only in the case of a mixed-content-mode element, and the HMRC
 * schemas never use those.
 * 
 * In situation B the error will be followed by the start event for the element
 * that raised the error.
 * 
 * So we schedule a fixup that will get canceled by an end-element event, or
 * processed and cleared by a start element.
 * 
 * @author Douglas Clinton
 * @since Apr 18, 2012
 * 
 */
public class CVCPatternValidDecoder extends AbstractXercesErrorDecoder {
    public boolean canDecode(final SAXParseException ex) {
        return ex.getMessage().startsWith("cvc-pattern-valid:");
    }

    public DocumentError decodeError(final SAXParseException ex, final ElementReference elementRef) {

        final String errorText = ex.getMessage();
        final int valueStart = errorText.indexOf("Value '") + "Value '".length();
        final int valueEnd = errorText.indexOf("'", valueStart);
        final String value = errorText.substring(valueStart, valueEnd);

        final DocumentError error;
        if (value.equals("")) {
            error = makeDocumentError(ex, elementRef, DocumentError.BLANK, value);
            getErrorPostProcessor().scheduleErrorForProcessing(error, FixupType.UntilEndElement);
        } else {
            error = makeDocumentError(ex, elementRef, DocumentError.FORMAT, value);
        }

        return error;
    }
}
