/**
 * $Id: CVCType313Decoder.java,v 1.2 2006/08/10 15:19:10 dec Exp $
 */
package com.gsl.sax.decoder.xerces.chris;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.xml.sax.SAXParseException;

import xmldoc.DocumentError;
import xmldoc.ElementReference;

import com.gsl.sax.decoder.xerces.AbstractXercesErrorDecoder;
import com.gsl.sax.decoder.xerces.ErrorDecoderState;

/**
 * A 3.1.3 can be generated with an incorrect location when the document uses an
 * empty tag (i.e. <tag/> ) on an element that cannot contain a blank value.
 * 
 * @author Douglas Clinton
 * @since Mar 27, 2006
 * 
 */
public class CVCType313Decoder extends AbstractXercesErrorDecoder {

    public static final String CVC_TYPE_3_1_3 = "cvc-type.3.1.3";

    /*
     * Case 878 - I'm now extracting just the element name, without the
     * namespace prefix, for comparison with the XPath location of the current
     * and previous errors to decide whether to fixup. This way we won't cause
     * the fixup if the schematron file and the document use different prefixes.
     */
    private final Pattern pattern = Pattern.compile("cvc-type.3.1.3: The value '(.*)' of element '(.*:)?(.+)' is not valid.");

    private final ErrorDecoderState state;

    public CVCType313Decoder(final ErrorDecoderState state) {
        this.state = state;
    }

    public boolean canDecode(final SAXParseException ex) {
        return ex.getMessage().startsWith(CVC_TYPE_3_1_3 + ":");
    }

    public DocumentError decodeError(final SAXParseException ex, final ElementReference elementRef) {
        DocumentError result;
        final Matcher matcher = pattern.matcher(ex.getMessage());
        if (matcher.matches()) {
            final String value = matcher.group(1);
            final String elementName = matcher.group(3);

            final DocumentError mostRecentError = state.getMostRecentError();

            result = makeDocumentError(ex, elementRef, CVC_TYPE_3_1_3, value);

            if (!mostRecentError.getElementReference().getXPath().endsWith(elementName)) {
                getErrorPostProcessor().scheduleErrorForProcessing(mostRecentError);
            }
            if (!elementRef.getXPath().endsWith(elementName)) {
                getErrorPostProcessor().scheduleErrorForProcessing(result);
            }
        } else {
            result = makeDocumentError(ex, elementRef, CVC_TYPE_3_1_3, null);
        }

        return result;
    }
}