/**
 * $Id: XercesExceptionDecoder.java,v 1.3 2006/08/10 15:19:10 dec Exp $
 */
package com.gsl.sax.decoder.xerces.erss;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.xml.sax.SAXParseException;

import xmldoc.DocumentError;
import xmldoc.ElementReference;
import xmldoc.sax.AbstractSAXExceptionDecoder;
import xmldoc.sax.DocumentErrorPostProcessor;
import xmldoc.sax.XPathLocationStack;

import com.gsl.sax.decoder.xerces.AbstractXercesErrorDecoder;
import com.gsl.sax.decoder.xerces.ErrorDecoderState;
import com.gsl.sax.decoder.xerces.XercesErrorDecoder;

/**
 * This is a utility class which provides a mechanism for converting a
 * SAXParseException to a DocumentError by decoding the text of the message from
 * the exception.
 * 
 * The aim of this is to eliminate the need for the error output stage of
 * processing (when the ProcessingErrors get converted to a form suitable for
 * returning to the user of the system) to know about Xerces and the format of
 * its errors. DocumentErrors have more abstract codes (such as "format" or
 * "value").
 * 
 * By decoupling the later stages from Xerces we also open up the possibility of
 * replacing Xerces with another parser.
 * 
 * @author Douglas Clinton
 * @since Mar 20, 2006
 * 
 */
public class XercesExceptionDecoder extends AbstractSAXExceptionDecoder {

    /**
     * This decoderMap holds the lookup table that lets us decide which kind of
     * decoder to use based on the beginning of the message text.
     * 
     * @link aggregation
     * @associates XercesErrorDecoder
     * 
     */
    private List<AbstractXercesErrorDecoder> decoders;

    private final ErrorDecoderState state = new ErrorDecoderState();

    public XercesExceptionDecoder() {
        initializeDecoders();
    }

    /**
     * Given an element reference and a SAXParserException this method will
     * generate a DocumentError which attempts to reflect the meaning of the
     * exception by inferring a DocumentError code. In addition, the
     * ElementReference held in the DocumentError may have been 'fixed-up', i.e.
     * changed to more accurately reflect the true position of the error since
     * Xerces reports many errors on the parent tag of the element where the
     * problem actually occured.
     */
    public DocumentError decodeException(final ElementReference elementRef, final SAXParseException ex) {
        DocumentError result = null;

        for (final Iterator<AbstractXercesErrorDecoder> iter = decoders.iterator(); iter.hasNext();) {
            final XercesErrorDecoder decoder = iter.next();
            if (decoder.canDecode(ex)) {
                result = decoder.decodeError(ex, elementRef);
                break;
            }
        }

        if (result == null) {
            result = makeDocumentError(ex, elementRef, DocumentError.UNKNOWN, null);
        }

        final String resultCode = result.getSubCode();
        if (!resultCode.equals(DocumentError.SUPPRESS) && !resultCode.equals(DocumentError.UNKNOWN)) {
            getState().setMostRecentError(result);
        }

        if (result != null) {
            result.setSAXErrorMessage(ex.getMessage());
        }

        return result;
    }

    private ErrorDecoderState getState() {
        return state;
    }

    private void initializeDecoders() {
        decoders = new ArrayList<AbstractXercesErrorDecoder>(20);

        decoders.add(new CVCType313Decoder(getState()));
        decoders.add(new CVCType311Decoder());
        decoders.add(new CVCPatternValidDecoder());
        decoders.add(new CVCBadFormatDecoder());
        decoders.add(new CVCEnumerationValidDecoder());
        decoders.add(new CVCComplexType24aDecoder(getState()));
        decoders.add(new CVCComplexType24bDecoder());
        decoders.add(new CVCComplexType22Decoder(getState()));
        decoders.add(new CVCMinInclusiveDecoder());
        decoders.add(new CVCMaxInclusiveDecoder());
        decoders.add(new CVCDataTypeValidDecoder());
        decoders.add(new CVCAttributeDecoder(getState()));
        decoders.add(new CVCComplexType322Decoder());
        decoders.add(new CVCMinLengthDecoder());
        decoders.add(new WellFormedTextOrMarkupDecoder());
        decoders.add(new BadEntityReferenceDecoder());
    }

    public DocumentError decodeException(final ElementReference currentElementRef, final XPathLocationStack locationStack,
            final SAXParseException ex) {
        throw new UnsupportedOperationException("implement me!");
    }

    public void setDocumentErrorPostProcessor(final DocumentErrorPostProcessor ruleRegistry) {
        // do nothing
    }
}
