/**
 * 
 */
package com.gsl.sax.xerces;

import java.util.LinkedList;
import java.util.logging.Logger;

import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLReaderFactory;

import com.gsl.logging.LoggerFactory;

/**
 * A large amount (~25%) of the startup overhead for submissions is down to
 * creating an XMLReader. This really impacts the performance of small
 * submission handling.
 * 
 * This pool helps to eliminate that overhead by letting the validator re-use
 * XMLReaders.
 * 
 * @author doug
 * 
 */
public class XMLReaderPool {

    private static final Logger logger = LoggerFactory.getLogger(XMLReaderPool.class.getName());

    private final LinkedList<XMLReader> pool = new LinkedList<XMLReader>();

    public XMLReader get() throws SAXException {
        XMLReader reader = null;

        synchronized (pool) {
            if (pool.isEmpty()) {
                reader = createReader(reader);
            } else {
                reader = pool.removeLast();
            }
        }

        return reader;
    }

    public void release(final XMLReader reader) {
        if (reader != null) {
            synchronized (pool) {
                pool.addFirst(reader);
            }
        } else {
            logger.warning("attempt to release null parser, ignoring");
        }
    }

    private XMLReader createReader(XMLReader reader) throws SAXException {
        try {
            reader = XMLReaderFactory.createXMLReader(DefaultXercesFeatureset.DEFAULT_PARSER_NAME);
        } catch (final SAXException e) {
            logger.severe("error: Unable to instantiate parser (" + DefaultXercesFeatureset.DEFAULT_PARSER_NAME + ")");
            e.printStackTrace();
            throw (e);
        }
        return reader;
    }
}
