package com.gsl.sax.xerces;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;

import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;

import xmldoc.ParserParams;
import xmldoc.util.ClasspathEntityResolver;

public class XSDIntrospector {

    private static final ParserParams parserParams = new ParserParams();

    static {
        parserParams.setDoSchemaValidation(false);
    }

    private String targetNamespace;

    public XSDIntrospector(final File file) {
        this(createFileInputStream(file));
    }

    private static FileInputStream createFileInputStream(final File file) {
        try {
            return new FileInputStream(file);
        } catch (final IOException e) {
            throw new RuntimeException(e);
        }
    }

    public XSDIntrospector(final InputStream in) {
        try {
            final XMLReader parser = XercesParserMaker.makeParser(parserParams);

            try {
                final SAXHandler saxHandler = new SAXHandler();
                parser.setContentHandler(saxHandler);
                parser.setErrorHandler(saxHandler);
                parser.setEntityResolver(saxHandler);
                parser.parse(new InputSource(new InputStreamReader(in, Charset.forName("UTF-8"))));
            } finally {
                XercesParserMaker.release(parser);
            }
        } catch (final SAXException done) {
        } catch (final RuntimeException e) {
            throw e;
        } catch (final Exception e) {
            throw new RuntimeException(e);
        }
    }

    public String getTargetNamespace() {
        return targetNamespace;
    }

    private class SAXHandler extends DefaultHandler {
        final String NAMESPACE_XML = "http://www.w3.org/XML/1998/namespace";

        final String[][] ENTITY_RESOLUTION_TABLE = {
                //
                { NAMESPACE_XML, "/xml.xsd" }, { "http://www.w3.org/2001/XMLSchema.dtd", "/XMLSchema.dtd" },
                { "/datatypes.dtd", "/datatypes.dtd" }, { "http://www.w3.org/2001/datatypes.dtd", "/datatypes.dtd" }
        //
        };

        final private ClasspathEntityResolver entityResolver = new ClasspathEntityResolver(ENTITY_RESOLUTION_TABLE);

        @Override
        public void startElement(final String namespaceURI, final String name, final String qName, final Attributes attributes)
                throws SAXException {

            if ("schema".equals(name)) {
                targetNamespace = attributes.getValue("targetNamespace");
                throw new SAXException("done");
            }
        }

        @Override
        public InputSource resolveEntity(final String publicId, String systemId) throws SAXException {
            if (systemId.startsWith("file://")) {
                systemId = systemId.substring(systemId.lastIndexOf("/"));
            }
            InputSource is = entityResolver.resolveEntity(publicId, systemId);
            if (is == null) {
                try {
                    is = super.resolveEntity(publicId, systemId);
                } catch (final SAXException saxException) {
                    throw saxException;
                } catch (final Exception e) {
                    /*
                     * Between Java 1.4 and Java 5 the signature for
                     * org.xml.sax.helpers.DefaultHandler.resolveEntity()
                     * changed to throw an IOException in addition to a
                     * SAXException. Because we need to build this library for
                     * both 1.4 (to support the PAYE validator) and 1.5 (to
                     * support the schematron validator) I am compensating for
                     * this by catching SAXExceptions and rethrowing them and
                     * converting any other exception into a RuntimeException.
                     * In Java 1.4 there won't be any other exception types
                     * thrown so this will be dead code, but in 1.5 this will
                     * re-throw an IOException as a RuntimeException.
                     */
                    throw new RuntimeException(e);
                }
            }
            if (is != null) {
                is.setSystemId(systemId);
            }
            return is;
        }
    }
}
