/*
 * Created on 27-Nov-2005
 * 
 * $Id: XercesParserMaker.java,v 1.1 2006/04/08 11:43:12 dec Exp $
 */
package com.gsl.sax.xerces;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

import org.xml.sax.SAXException;
import org.xml.sax.SAXNotRecognizedException;
import org.xml.sax.SAXNotSupportedException;
import org.xml.sax.XMLReader;

import xmldoc.ParserParams;

import com.gsl.logging.LoggerFactory;

/**
 * @author Douglas Clinton
 * @since 27-Nov-2005
 */
public class XercesParserMaker implements DefaultXercesFeatureset {

    private static final Logger log = LoggerFactory.getLogger(XercesParserMaker.class.getName());

    private static XMLReaderPool readerPool = new XMLReaderPool();

    public static final String NAMESPACE_PREFIXES_PROPERTY = "namespacePrefixes";

    private static final String VALIDATION_PROPERTY = "validation";

    private static final String SCHEMA_VALIDATION_PROPERTY = "schemaValidation";

    /**
     * We provide this global object to Xerces for it to use as a cache for the
     * schemas. The first time a schema is loaded by Xerces it will be cached in
     * this pool and re-used on subsequent parses. This greatly speeds the
     * processing time for the second and subsequent parses.
     */
    private static final XMLGrammarPoolMap GRAMMAR_POOL_MAP = new XMLGrammarPoolMapImpl();

    public static XMLReader makeParser(final ParserParams parserParams) throws SAXException {
        return makeParser(parserParams, "", null);
    }

    /**
     * It can be useful at times (specifically in the PAYE development) for the
     * creator of the parser to be able to select a grammar pool cache based on
     * a key. The specific case we implemented this for dealt with the issue
     * that the PAYE namespaces for tax years 0405 and 0506 are the same, but
     * the schemas are different (in subsequent years the namespaces are
     * versioned.) Without the grammarPoolKey, and given that many parsers may
     * be created within the lifetime of a single JVM, processing an 0405
     * submission would result in the schema for 0405 being cached and then used
     * by subsequent processing of an 0506 submission, which would be wrong.
     * Adding in the key lets us distinguish between tax years, but I have made
     * the key an object so that this class is more generally useful and not
     * tied to the PAYE system.
     * 
     * @param parserFeatures
     * @param entityResolutionTable
     * @param grammarPoolKey
     * @return
     * @throws SAXException
     */
    public static XMLReader makeParser(final ParserParams parserParams, final String[][] entityResolutionTable,
            final Object grammarPoolKey) throws SAXException {
        return makeParser(parserParams, prepareSchemaLocations(entityResolutionTable), grammarPoolKey);
    }

    public static void flushGrammarPool(final Object grammarPoolKey) {
        GRAMMAR_POOL_MAP.flush(grammarPoolKey);
    }

    public static XMLReader makeParser(final ParserParams parserParams, final String externalSchemaLocations,
            final Object grammarPoolKey) throws SAXException {
        final XMLReader parser = readerPool.get();

        Map<String, Boolean> parserFeatures = null;
        if (!parserParams.getDoSchemaValidation()) {
            parserFeatures = new HashMap<String, Boolean>();
            parserFeatures.put(XercesParserMaker.VALIDATION_PROPERTY, Boolean.FALSE);
            parserFeatures.put(XercesParserMaker.SCHEMA_VALIDATION_PROPERTY, Boolean.FALSE);
        }

        // set parser features
        final boolean namespaces = DEFAULT_NAMESPACES;
        boolean namespacePrefixes = DEFAULT_NAMESPACE_PREFIXES;
        boolean validation = DEFAULT_VALIDATION;
        boolean schemaValidation = DEFAULT_SCHEMA_VALIDATION;
        final boolean schemaFullChecking = DEFAULT_SCHEMA_FULL_CHECKING;
        final boolean dynamicValidation = DEFAULT_DYNAMIC_VALIDATION;

        if (parserFeatures != null) {
            final Boolean suppliedNamespacePrefixes = parserFeatures.get(NAMESPACE_PREFIXES_PROPERTY);
            if (suppliedNamespacePrefixes != null) {
                namespacePrefixes = suppliedNamespacePrefixes.booleanValue();
            }
            final Boolean suppliedValidation = parserFeatures.get(VALIDATION_PROPERTY);
            if (suppliedValidation != null) {
                validation = suppliedValidation.booleanValue();
            }
            final Boolean suppliedSchemaValidation = parserFeatures.get(SCHEMA_VALIDATION_PROPERTY);
            if (suppliedSchemaValidation != null) {
                schemaValidation = suppliedSchemaValidation.booleanValue();
            }
        }

        try {
            parser.setProperty(EXTERNAL_SCHEMALOCATION_PROPERTY_ID, externalSchemaLocations);
        } catch (final SAXException e) {
            log.warning("Parser does not support property (" + EXTERNAL_SCHEMALOCATION_PROPERTY_ID + ")");
        }
        try {
            parser.setProperty(GRAMMAR_POOL_PROPERTY_ID, GRAMMAR_POOL_MAP.getXMLGrammarPool(grammarPoolKey));
        } catch (final SAXException e) {
            log.warning("Parser does not support property (" + GRAMMAR_POOL_PROPERTY_ID + ")");
        }
        try {
            parser.setFeature(NAMESPACES_FEATURE_ID, namespaces);
        } catch (final SAXException e) {
            log.warning("Parser does not support feature (" + NAMESPACES_FEATURE_ID + ")");
        }
        try {
            parser.setFeature(NAMESPACE_PREFIXES_FEATURE_ID, namespacePrefixes);
        } catch (final SAXException e) {
            log.warning("Parser does not support feature (" + NAMESPACE_PREFIXES_FEATURE_ID + ")");
        }
        try {
            parser.setFeature(VALIDATION_FEATURE_ID, validation);
        } catch (final SAXException e) {
            log.warning("Parser does not support feature (" + VALIDATION_FEATURE_ID + ")");
        }
        try {
            parser.setFeature(SCHEMA_VALIDATION_FEATURE_ID, schemaValidation);
        } catch (final SAXNotRecognizedException e) {
            log.warning("Parser does not support feature (" + SCHEMA_VALIDATION_FEATURE_ID + ")");
        } catch (final SAXNotSupportedException e) {
            log.warning("Parser does not support feature (" + SCHEMA_VALIDATION_FEATURE_ID + ")");
        }
        try {
            parser.setFeature(SCHEMA_FULL_CHECKING_FEATURE_ID, schemaFullChecking);
        } catch (final SAXNotRecognizedException e) {
            log.warning("Parser does not support feature (" + SCHEMA_FULL_CHECKING_FEATURE_ID + ")");
        } catch (final SAXNotSupportedException e) {
            log.warning("Parser does not support feature (" + SCHEMA_FULL_CHECKING_FEATURE_ID + ")");
        }
        try {
            parser.setFeature(DYNAMIC_VALIDATION_FEATURE_ID, dynamicValidation);
        } catch (final SAXNotRecognizedException e) {
            log.warning("Parser does not support feature (" + DYNAMIC_VALIDATION_FEATURE_ID + ")");
        } catch (final SAXNotSupportedException e) {
            log.warning("Parser does not support feature (" + DYNAMIC_VALIDATION_FEATURE_ID + ")");
        }
        return parser;
    }

    public static void release(final XMLReader parser) {
        readerPool.release(parser);
    }

    /**
     * Xerces needs a string containing a list of pairs of schema identifiers.
     * The first string of each pair is the namespace identifier as used in the
     * XML element start tag and the second is the identifer to which it should
     * be mapped for entity resolution. That second string will be passed to the
     * resolveEntity() method of the DocumentHandler.
     * 
     * In practice, we want both of these to be the same so we read the table
     * and prepare an output string which just contains each table entry
     * duplicated and separated by a space.
     * 
     * @since 27-Nov-2005
     */
    private static String prepareSchemaLocations(final String[][] entityResolutionTable) {
        // use default schemas if no schemas supplied
        final String[] schemas = new String[entityResolutionTable.length];
        for (int i = 0; i < entityResolutionTable.length; i++) {
            final String[] entry = entityResolutionTable[i];
            schemas[i] = entry[0];
        }

        String externalSchemaLocations = "";

        for (final String element : schemas) {
            externalSchemaLocations += element + " " + element + " ";
        }
        return externalSchemaLocations;
    }

}
