package com.gsl.util;

import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;

/**
 * A map of [comparable field, direction of comparison] ==> {@code Comparator}.
 * Since comparators are stateless, there is no point creating more than one of
 * each class. A static comparator map is a convenient container for storing
 * ready-made comparators.
 * 
 * @author rhys
 */

public class ComparatorMap<T> {

    public static final int FORWARD_COMPARISON = 0, REVERSE_COMPARISON = 1;

    private final Map<ComparatorIdAndDirection, Comparator<T>> map = new HashMap<ComparatorIdAndDirection, Comparator<T>>();

    public void add(final int id, final int direction, final Comparator<T> comparator) {
        final ComparatorIdAndDirection key = new ComparatorIdAndDirection(id, direction);
        map.put(key, comparator);
    }

    public Comparator<T> get(final int id, final int direction) {
        final ComparatorIdAndDirection key = new ComparatorIdAndDirection(id, direction);
        return map.get(key);
    }
}

class ComparatorIdAndDirection {
    final int id;

    final int direction;

    final int hashCode;

    ComparatorIdAndDirection(final int id, final int direction) {
        this.id = id;
        this.direction = direction;
        this.hashCode = calculateHashCode();
    }

    private int calculateHashCode() {
        int result = HashCode.SEED;
        result = HashCode.hash(result, id);
        result = HashCode.hash(result, direction);
        return result;
    }

    @Override
    public int hashCode() {
        return hashCode;
    }

    @Override
    public boolean equals(final Object o) {
        if (o == this)
            return true;
        if (o == null || o.getClass() != getClass())
            return false;
        final ComparatorIdAndDirection other = (ComparatorIdAndDirection) o;
        return ((id == other.id) && (direction == other.direction));
    }
}
