package com.gsl.util;

import java.io.Serializable;
import java.util.Comparator;
import java.util.Date;

import com.gsl.util.version.Version;
import com.gsl.util.version.Versioned;

/**
 * A file descriptor provides very basic information about a file, such as
 * filepath, filename and last modified timestamp, without storing a reference
 * to the file itself.
 * <p>
 * 
 * This class is completely independent of any file implementation (such as
 * java.io.File) and any file system. It does not know and should not know that
 * any particular file implementation or file system exists. The file it
 * describes is any abstract file in any file system, and the filepath it
 * contains is any abstract filepath.
 * <p>
 * 
 * Since it is immutable, a file descriptor is completely decoupled from changes
 * to the underlying file (if the file even exists). It is also versioned and
 * provides useful comparators.
 * 
 * @author rhys
 */

public class FileDescriptor implements Serializable, Versioned, Comparable<FileDescriptor> {

    public static final int FILEPATH_ID = 0, LAST_MODIFIED_ID = 1;

    private static ComparatorMap<FileDescriptor> COMPARATOR_MAP = new ComparatorMap<FileDescriptor>();

    private static Comparator<FileDescriptor> FILEPATH_LAST_MODIFIED_COMPARATOR;

    static {
        final CompositeComparatorBuilder<FileDescriptor> builder = new CompositeComparatorBuilder<FileDescriptor>();
        builder.add(getComparator(FILEPATH_ID, ReversibleComparator.FORWARD_COMPARISON));
        builder.add(getComparator(LAST_MODIFIED_ID, ReversibleComparator.FORWARD_COMPARISON));
        FILEPATH_LAST_MODIFIED_COMPARATOR = builder.makeCompositeComparator();
    }

    /*
     * Q: Isn't this lazy static initialization potentially thread-unsafe? A:
     * No. Comparators are stateless, and no adverse effect will occur if a race
     * condition happens and two comparators of the same class momentarily
     * co-exist.
     */
    public static Comparator<FileDescriptor> getComparator(final int id, final int direction) {
        Comparator<FileDescriptor> comparator = COMPARATOR_MAP.get(id, direction);

        if (comparator == null) {
            switch (id) {
            case FILEPATH_ID:
            default:
                comparator = new FilepathComparator(direction);
                break;

            case LAST_MODIFIED_ID:
                comparator = new LastModifiedComparator(direction);
                break;

            }

            COMPARATOR_MAP.add(id, direction, comparator);
        }
        return comparator;
    }

    private final String filepath;

    private final String filename;

    private final Date lastModified;

    private final Version version;

    private final int hashCode;

    /**
     * Constructs a timestamped versioned file descriptor.
     * 
     * @param filepath
     *            path to the file i.e. directory path followed by name of the
     *            file.
     * @param lastModified
     *            last modified date of the file.
     * @param version
     *            the version of the file, expressed as an array.
     */
    public FileDescriptor(final String filepath, final Date lastModified, final Version version) {
        this.filepath = filepath;
        final int filepathSeparatorPos = filepath.lastIndexOf(System.getProperty("file.separator"));
        this.filename = (filepathSeparatorPos != -1) ? filepath.substring(filepathSeparatorPos + 1) : "";
        this.lastModified = lastModified;
        this.version = version;

        this.hashCode = calculateHashCode();
    }

    public FileDescriptor(final String filepath, final String filename) {
        this.filepath = filepath;
        this.filename = filename;
        this.lastModified = null;
        this.version = VERSION_IF_NONE_SPECIFIED;

        this.hashCode = calculateHashCode();
    }

    /**
     * Constructs a timestamped file descriptor.
     * 
     * @param filepath
     *            path to the file i.e. directory path followed by name of the
     *            file.
     * @param lastModified
     *            last modified date of the file.
     */
    public FileDescriptor(final String filepath, final Date lastModified) {
        this(filepath, lastModified, VERSION_IF_NONE_SPECIFIED);
    }

    /**
     * Constructs a file descriptor without timestamp.
     * 
     * @param filepath
     *            path to the file i.e. directory path followed by name of the
     *            file.
     */
    public FileDescriptor(final String filepath) {
        this(filepath, (Date) null);
    }

    private int calculateHashCode() {
        int result = HashCode.SEED;
        result = HashCode.hash(result, filepath);
        result = HashCode.hash(result, lastModified);
        result = HashCode.hash(result, version);
        return result;
    }

    public String getFilepath() {
        return filepath;
    }

    public String getFilename() {
        return filename;
    }

    public String getFilenameExtension() {
        final int periodPos = filename.lastIndexOf('.');
        final String fileNameExtension = (periodPos != -1) ? filename.substring(periodPos + 1) : "";
        return fileNameExtension;
    }

    public Date getLastModifiedDate() {
        return lastModified;
    }

    public Version getVersion() {
        return version;
    }

    @Override
    public String toString() {
        return filepath + " (" + lastModified + ", " + version.toString() + ")";
    }

    @Override
    public int hashCode() {
        return hashCode;
    }

    @Override
    public boolean equals(final Object rhs) {
        if (rhs == this)
            return true;

        boolean equals = false;

        if (rhs != null && rhs.getClass() == FileDescriptor.class) {
            final FileDescriptor fd = (FileDescriptor) rhs;

            equals = this.filepath.equals(fd.filepath) && ObjectUtil.nullSafeEquals(this.lastModified, fd.lastModified)
                    && this.version.equals(fd.version);
        }

        return equals;
    }

    /*
     * There is arguably no single natural ordering of FileDescriptors. This
     * Comparable implementation is kept for compatibility with existing code.
     */
    public int compareTo(final FileDescriptor o) {
        return FILEPATH_LAST_MODIFIED_COMPARATOR.compare(this, o);
    }
}

class FilepathComparator implements ReversibleComparator<FileDescriptor> {
    private final int multiplier;

    FilepathComparator(final int direction) {
        this.multiplier = (direction == REVERSE_COMPARISON) ? -1 : 1;
    }

    public int compare(final FileDescriptor fd1, final FileDescriptor fd2) {
        return fd1.getFilepath().compareTo(fd2.getFilepath()) * multiplier;
    }
}

class LastModifiedComparator implements ReversibleComparator<FileDescriptor> {
    private final int multiplier;

    public LastModifiedComparator(final int direction) {
        this.multiplier = (direction == REVERSE_COMPARISON) ? -1 : 1;
    }

    public int compare(final FileDescriptor fd1, final FileDescriptor fd2) {
        return ObjectUtil.nullSafeCompare(fd1.getLastModifiedDate(), fd2.getLastModifiedDate()) * multiplier;
    }
}
