/**
 * $Id: $
 */
package com.gsl.util.classlookup;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.*;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * 
 * Derived from, ClassPathTestCollector from JUnit 3.8
 * <p>
 * Override the isMatchingClass() method to provide specific filtering.
 * 
 * @author Doug Clinton
 * 
 */
public class ClassPathClassCollector implements ClassNameCollector {
    static final int SUFFIX_LENGTH = ".class".length();

    /**
     * Part of the fix for the JRockit issue that can crop up in the
     * gatherFiles() method below. Declare this as a member far, always set to
     * false because if I just put 'false' into the if statement below then I
     * get a warning about dead code that I can't suppress.
     */
    private final boolean catchFileNotFoundFix = false;

    public Collection<String> collectClassNames() {
        final String classPath = System.getProperty("java.class.path");
        final Map<String, String> result = collectFilesInPath(classPath);
        return result.values();
    }

    public Map<String, String> collectFilesInPath(final String classPath) {
        final Map<String, String> result = collectFilesInRoots(splitClassPath(classPath));
        return result;
    }

    Map<String, String> collectFilesInRoots(final List<String> roots) {
        final Map<String, String> result = new Hashtable<String, String>(100);
        for (final String nextRoot : roots) {
            if (nextRoot.endsWith(".jar")) {
                gatherJarFileEntries(nextRoot, result);
            } else {
                gatherFiles(new File(nextRoot), "", result);
            }
        }
        return result;
    }

    private void gatherJarFileEntries(final String classRoot, final Map<String, String> result) {
        try {
            final JarFile jarFile = new JarFile(classRoot);
            final Enumeration<JarEntry> e = jarFile.entries();
            while (e.hasMoreElements()) {
                final JarEntry entry = e.nextElement();
                final String entryName = entry.getName();
                if (isMatchingClass(entryName)) {
                    final String className = classNameFromFile(entryName);
                    result.put(className, className);
                }
            }
        } catch (final IOException e) {
            // ignore
        }
    }

    void gatherFiles(final File classRoot, final String classFileName, final Map<String, String> result) {
        final File thisRoot = new File(classRoot, classFileName);
        if (thisRoot.isFile()) {
            if (isMatchingClass(classFileName)) {
                final String className = classNameFromFile(classFileName);
                result.put(className, className);
            }
            return;
        } else if (thisRoot.isDirectory()) {
            try {
                final String[] contents = thisRoot.list();

                if (contents != null) {
                    for (final String element : contents) {
                        gatherFiles(classRoot, classFileName + File.separatorChar + element, result);
                    }
                }

                // keep compiler happy - see comment below about JRockit. The
                // compiler will complain about the catch if we don't do this
                // because nothing within the try is declared to throw the
                // exception.
                if (catchFileNotFoundFix) {
                    throw new FileNotFoundException();
                }

            } catch (final FileNotFoundException t) {
                // JRockit on Windows (the WebLogic 8 environment that Aspire
                // deploy into for testing) throws a FileNotFoundException out
                // of File.list() under some circumstances, even though the Java
                // API spec says that File.list() does not throw any exceptions.
                // That's why we have to catch it as a Throwable.
                //
                // Just ignore it. Nothing to see here, move on
            }
        }
    }

    List<String> splitClassPath(final String classPath) {
        final List<String> result = new ArrayList<String>();
        final String separator = System.getProperty("path.separator");
        final StringTokenizer tokenizer = new StringTokenizer(classPath, separator);
        while (tokenizer.hasMoreTokens()) {
            result.add(tokenizer.nextToken());
        }
        return result;
    }

    /**
     * Override this method to provide specific filtering.
     * 
     * @param classFileName
     * @return
     */
    protected boolean isMatchingClass(final String classFileName) {
        return classFileName.endsWith(".class") && classFileName.indexOf('$') < 0;
    }

    protected String classNameFromFile(final String classFileName) {
        // convert /a/b.class to a.b
        final String s = classFileName.substring(0, classFileName.length() - SUFFIX_LENGTH);
        String s2 = s.replace(File.separatorChar, '.');
        /*
         * HACK: If we're on Unix then the previous line will already have
         * replaced '/' with '.' but on windows it will have replaced '\'. If
         * the file comes from a jar file, however, then the separarator is
         * always '/' so do this second replacement to catch that.
         */
        s2 = s2.replace('/', '.');
        if (s2.startsWith(".")) {
            return s2.substring(1);
        }
        return s2;
    }
}
