/**
 * $Id: $
 */
package com.gsl.util.classlookup;

import java.lang.reflect.Constructor;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import com.gsl.logging.LoggerFactory;

/**
 * This implementation searches in a base package to find all classes that
 * implement a given interface.
 * 
 * Matching classes are cached so that another lookup within the JVM for the
 * same base package and interface will return the cached set. We have assumed
 * that the set of classes avaialble will not change during the lifetime of the
 * JVM.
 * 
 * @author Doug Clinton
 * 
 */
public class InterfaceInstanceCollector extends PackageClassCollector {

    private static final Logger LOGGER = LoggerFactory.getLogger("class.collector");

    private final Class<?> implementsInterface;

    private final String stringName;

    public InterfaceInstanceCollector(final String basePackageName, final Class<?> implementsInterface) {
        super(basePackageName);

        stringName = "InterfaceInstanceCollector [basePackageName='" + basePackageName + "', interfaceCollector="
                + implementsInterface.getName() + "]";

        this.implementsInterface = implementsInterface;
    }

    @Override
    protected boolean isMatchingClass(final String classFileName) {
        boolean result = false;

        if (super.isMatchingClass(classFileName)) {
            final String className = classNameFromFile(classFileName);
            try {
                final Class<?> clazz = Class.forName(className);
                if (Modifier.isPublic(clazz.getModifiers()) && !Modifier.isAbstract(clazz.getModifiers())) {
                    result = checkImplementsInterface(clazz);
                }
            } catch (final ClassNotFoundException e) {
                // do nothing - if we can't find the class for this name
                // then we don't want to include it in the results.
                LOGGER.info("class not found attempting to load: " + className + ": ");
            } catch (final SecurityException se) {
                // do nothing, will return false
                LOGGER.info("security exception attempting to load: " + className + ": ");
            } catch (final Throwable t) {
                LOGGER.severe("failed to load class: " + className + ": " + t.getMessage());
            }
        }
        return result;
    }

    @Override
    public String toString() {
        return stringName;
    }

    /**
     * Class must be public, non-abstract, implement the requested interface and
     * have a public, no-args constructor.
     * 
     * @param clazz
     * @return
     */
    private boolean checkImplementsInterface(final Class<?> clazz) {
        boolean result = false;
        final List<Class<?>> interfaces = getInterfaces(clazz);
        if (interfaces.contains(implementsInterface)) {
            final Constructor<?>[] constructors = clazz.getConstructors();
            for (final Constructor<?> constructor : constructors) {
                if (constructor.getParameterTypes().length == 0 && Modifier.isPublic(constructor.getModifiers())) {
                    result = true;
                }
            }
        }
        return result;
    }

    private List<Class<?>> getInterfaces(final Class<?> clazz) {
        final List<Class<?>> result = new ArrayList<Class<?>>();

        Class<?> classToCheck = clazz;
        while (classToCheck != null) {
            final Class<?>[] interfaces = classToCheck.getInterfaces();
            for (final Class<?> element : interfaces) {
                result.add(element);
            }
            classToCheck = classToCheck.getSuperclass();
        }
        return result;
    }
}
