/*
 * Created on Apr 15, 2006
 *
 * $Id: MethodLocatorCache.java,v 1.5 2006/05/26 08:34:51 dec Exp $
 */
package com.gsl.util.reflection;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import com.gsl.util.cache.AbstractCache;
import com.gsl.util.collections.NestedHashMapTwoKeyMap;
import com.gsl.util.collections.TwoKeyMap;

/**
 * Provides a cache for method lookups.
 * 
 * We use a null placeholder in the cache for methods which were not found using
 * the chained locator so that subsequent lookups for the method don't waste
 * time trying, and failing, again.
 * 
 * @author douglas
 * @since Apr 15, 2006
 */
public class MethodLocatorCache extends AbstractCache implements MethodLocator {

    /**
     * @directed true
     */

    private final MethodLocator chainedMethodLocator;

    /**
     * @link aggregation
     * @associates MethodLocatorCacheEntry
     */
    private final Map<ClassLoader, TwoKeyMap<Class<?>, String, MethodLocatorCacheEntry>> cache = new HashMap<ClassLoader, TwoKeyMap<Class<?>, String, MethodLocatorCacheEntry>>();

    /**
     * Create a cache which will use the chainedMethodLocator as the source for
     * methods which are not already cached.
     */
    public MethodLocatorCache(final MethodLocator chainedMethodLocator) {
        this.chainedMethodLocator = chainedMethodLocator;
    }

    /**
     * @return the method on the given class with methodName. If the method is
     *         not found in the cache then this object will defer to the chained
     *         method locator to find it. Returns null if the method was not
     *         found.
     */
    public Method locateMethod(final Class<?> clazz, final String methodName) {
        synchronized (cache) {
            final ClassLoader classloader = clazz.getClassLoader();

            TwoKeyMap<Class<?>, String, MethodLocatorCacheEntry> map = cache.get(classloader);
            if (map == null) {
                map = new NestedHashMapTwoKeyMap<Class<?>, String, MethodLocatorCacheEntry>();
                cache.put(classloader, map);
            }

            MethodLocatorCacheEntry entry = map.get(clazz, methodName);

            if (entry == null) {
                miss();
                final Method method = chainedMethodLocator.locateMethod(clazz, methodName);
                if (method != null) {
                    entry = new MethodLocatorCacheEntry(method);
                    map.put(clazz, methodName, entry);
                } else {
                    entry = MethodLocatorCacheEntry.NullPlaceholder;
                    map.put(clazz, methodName, entry);
                }
            } else {
                hit();
            }
            return entry.getMethod();
        }
    }

    @Override
    public void flush() {
        synchronized (cache) {
            cache.clear();
        }
    }

    public void flush(final ClassLoader classloader) {
        synchronized (cache) {
            final TwoKeyMap<Class<?>, String, MethodLocatorCacheEntry> map = cache.get(classloader);
            if (map != null) {
                map.clear();
            }
        }
    }
}
