/*
 * Created on Apr 15, 2006
 *
 * $Id: SetterMethodLocator.java,v 1.3 2006/05/26 08:35:39 dec Exp $
 */
package com.gsl.util.reflection;

import java.lang.reflect.Method;

/**
 * Locates a setter based on a property name using standard java bean naming
 * conventions, i.e. a property of <em>foo</em> would have a setter method
 * called <em>setFoo</em>. The choice of paramter and return types is made by
 * the chained locator. This class merely deals with generating the right method
 * name.
 * 
 * @author douglas
 * @since Apr 15, 2006
 */
public class SetterMethodLocator implements MethodLocator {

    /**
     * @directed true
     * @label chainedLocator
     */

    private final MethodLocator chainedMethodLocator;

    /**
     * Setters usually take one parameter and have void return type. This
     * convenience constructor will set up a locator which looks for that.
     * 
     * @param argType
     *            type of the parameter of the property being set.
     */
    public SetterMethodLocator(final Class<?> argType) {
        this(new Class[] { argType });
    }

    public SetterMethodLocator(final Class<?>[] argTypes) {
        this.chainedMethodLocator = new BasicMethodLocator(argTypes, Void.TYPE);
    }

    /**
     * Once the method name is determined from the property name, the actual
     * method lookup is delegated off to the chained locator.
     */
    public SetterMethodLocator(final MethodLocator chainedMethodLocator) {
        this.chainedMethodLocator = chainedMethodLocator;
    }

    /**
     * The propertyName is converted to a setter method name using standard java
     * bean conventions and that setter is looked for on the chained locator.
     */
    public Method locateMethod(final Class<?> clazz, final String propertyName) {
        final String setterName = makeSetterName(propertyName);
        return chainedMethodLocator.locateMethod(clazz, setterName);
    }

    /**
     * Given a property name, construct the name of a setter method according to
     * the standard convention for java beans. I.e. For a property named
     * <i>propertyName </i> we will make the setter name <i>setPropertyName
     * </i>.
     * 
     * @param name
     *            name of the property to convert to a setter name
     * @return the name of a setter method for the property.
     * @throws IllegalArgumentException
     *             if the <i>name </i> is null or a blank string.
     */
    private static String makeSetterName(final String name) throws IllegalArgumentException {
        String result = "";
        if (name == null) {
            throw new IllegalArgumentException("name should not be null in makeSetterName");
        }

        if (name.trim().length() == 0) {
            throw new IllegalArgumentException("name should not be blank in makeSetterName");
        }

        final String firstChar = name.substring(0, 1).toUpperCase();
        final String rest = name.substring(1);

        result = "set" + firstChar + rest;

        return result;
    }
}
