package ericResponse.output;

import java.io.Writer;
import java.util.List;

import processingError.ProcessingError;

import com.gsl.docValidator.PostValidationDelegate;

import ericResponse.output.error.EricResponseError;
import ericResponse.output.error.EricResponseErrorFactory;
import ericResponse.output.translator.TranslationContext;
import ericResponse.output.translator.genericerror.GenericErrorMapper;
import ericResponse.output.translator.mapper.ErrorMapper;

/**
 * This class provides a base PostValidationDelegate implementation suitable for
 * subclassing by validator implementations that will be incorporating an ERIC
 * response output stage. It is responsible for bringing together the various
 * (mandatory and optional) ERIC response components.
 * 
 * @author jesse
 */
public abstract class EricResponsePostValidationDelegate implements PostValidationDelegate {

    private final EricResponseParams responseParams;

    private final Writer output;

    private EricResponseErrorMapper errorMapper;

    private EricResponseGenerator outputGenerator;

    public EricResponsePostValidationDelegate(final EricResponseParams responseParams, final Writer output) {
        this.responseParams = responseParams;
        this.output = output;
    }

    protected EricResponseErrorMapper getErrorMapper() {
        if (errorMapper == null) {
            errorMapper = new EricResponseErrorMapper(responseParams, createEricResponseErrorFactory());
        }
        return errorMapper;
    }

    protected EricResponseGenerator getOutputGenerator() {
        if (output != null) {
            if (outputGenerator == null) {
                outputGenerator = new EricResponseGenerator(responseParams);
            }
            return outputGenerator;
        }
        return null;
    }

    public void addResponseGenerator(final ResponseElementGenerator generator) {
        if (output != null) {
            getOutputGenerator().addResponseGenerator(generator);
        }
    }

    public void addErrorMapper(final ErrorMapper mapper) {
        getErrorMapper().addErrorMapper(mapper);
    }

    public void addErrorMappers(final String basePackageName) {
        getErrorMapper().addErrorMappers(basePackageName);
    }

    public void addGenericMapper(final GenericErrorMapper mapper) {
        getErrorMapper().addGenericMapper(mapper);
    }

    public void addGenericMappers(final String basePackageName) {
        getErrorMapper().addGenericMappers(basePackageName);
    }

    public void addErrorFilter(final EricErrorFilter filter) {
        getErrorMapper().addEricErrorFilter(filter);
    }

    public List<EricResponseError> mapErrors(final List<ProcessingError> processingErrors) {
        final List<EricResponseError> ericErrors = getErrorMapper().mapProcessingErrors(processingErrors,
                createTranslationContext());

        return ericErrors;
    }

    protected void generateResponse(final List<EricResponseError> ericErrors) {
        if (output != null) {
            getOutputGenerator().generateResponse(ericErrors, output);
        }
    }

    protected TranslationContext createTranslationContext() {
        // defaults to null
        return null;
    }

    protected EricResponseErrorFactory createEricResponseErrorFactory() {
        // causes default to be used
        return null;
    }

    public EricResponseError mapError(final ProcessingError pe) {
        return getErrorMapper().mapProcessingError(pe, createTranslationContext());
    }
}
