/**
 * $Id: GovTalkMessage.java,v 1.17 2006/04/20 08:11:53 dec Exp $
 */
package ericResponse.output;

import java.util.HashMap;
import java.util.Map;

import org.jdom.Attribute;
import org.jdom.Element;
import org.jdom.Namespace;

import govtalk.header.GovTalkHeaderDetails;

/**
 * @author Douglas Clinton
 * @since Jan 27, 2006
 * 
 */
public class GovTalkMessage {

    public static final String GOVTALK_ENVELOPE_NS = "http://www.govtalk.gov.uk/CM/envelope";

    public static final String GOVTALK_ENVELOPE_LOCATION_V1 = "http://www.govtalk.gov.uk/CM/envelope-v1.xsd";

    public static final String GOVTALK_ENVELOPE_LOCATION_V2 = "http://www.govtalk.gov.uk/CM/envelope-v2.xsd";

    public static final String XSI_NAMESPACE = "http://www.w3.org/2001/XMLSchema-instance";

    private static final Namespace xsiNamespace = Namespace.getNamespace("xsi", XSI_NAMESPACE);

    private final Element rootElement;

    private final Element bodyElement;

    private static final Map<String, String> envelopeLocation = new HashMap<String, String>();

    static {
        envelopeLocation.put("1.0", GOVTALK_ENVELOPE_LOCATION_V1);
        envelopeLocation.put("2.0", GOVTALK_ENVELOPE_LOCATION_V2);
    }

    /**
     * Use the generate method to create one of these.
     * 
     * @param rootElement
     * @param bodyElement
     */
    private GovTalkMessage(final Element rootElement, final Element bodyElement) {
        this.rootElement = rootElement;
        this.bodyElement = bodyElement;
    }

    public Element getBodyElement() {
        return bodyElement;
    }

    public Element getRootElement() {
        return rootElement;
    }

    public static GovTalkMessage generate(final GovTalkHeaderDetails submissionHeader, final int errorCount,
            final EricResponseParams params) {

        if (params.getGenerateGovTalkEnvelope()) {
            final Element rootElement = new Element("GovTalkMessage", GOVTALK_ENVELOPE_NS);
            rootElement.addNamespaceDeclaration(xsiNamespace);
            {
                String gatewayEnvelopeVersion = submissionHeader.getGatewayEnvelopeVersion();
                if (params.getOverrideGatewayEnvelopeVersion() != null) {
                    gatewayEnvelopeVersion = params.getOverrideGatewayEnvelopeVersion();
                }

                final Attribute schemaLocationAttribute = new Attribute("schemaLocation", GOVTALK_ENVELOPE_NS + " "
                        + envelopeLocation.get(gatewayEnvelopeVersion), xsiNamespace);

                rootElement.setAttribute(schemaLocationAttribute);

                final Element envelopeVersion = new Element("EnvelopeVersion", GOVTALK_ENVELOPE_NS);

                envelopeVersion.addContent(gatewayEnvelopeVersion);
                rootElement.addContent(envelopeVersion);
            }

            writeHeader(rootElement, errorCount, submissionHeader, params);

            final Element govTalkDetails = generateGovTalkDetails(rootElement, errorCount, params, submissionHeader);
            rootElement.addContent(govTalkDetails);

            final Element body = generateBody(rootElement);
            return new GovTalkMessage(rootElement, body);
        } else {
            final Element body = generateBody(null);
            return new GovTalkMessage(body, body);
        }

    }

    private static Element generateGovTalkDetails(final Element rootElement, final int errorCount, final EricResponseParams params,
            final GovTalkHeaderDetails header) {

        final Element result = new Element("GovTalkDetails", GOVTALK_ENVELOPE_NS);

        final Element keys = new Element("Keys", GOVTALK_ENVELOPE_NS);
        keys.addNamespaceDeclaration(xsiNamespace);
        result.addContent(keys);
        if (params.includeKeys()) {
            final Element taxOfficeNumber = new Element("Key", GOVTALK_ENVELOPE_NS);
            taxOfficeNumber.setAttribute(new Attribute("Type", "TaxOfficeNumber"));
            taxOfficeNumber.addContent(header.getTaxOfficeNumber());
            keys.addContent(taxOfficeNumber);

            final Element taxOfficeRef = new Element("Key", GOVTALK_ENVELOPE_NS);
            taxOfficeRef.setAttribute(new Attribute("Type", "TaxOfficeReference"));
            taxOfficeRef.addContent(header.getTaxOfficeReference());
            keys.addContent(taxOfficeRef);

        }

        if (errorCount > 0 && params.areGovTalkErrorsRequired()) {
            // Generate some extra info
            final Element govTalkErrors = new Element("GovTalkErrors", GOVTALK_ENVELOPE_NS);
            result.addContent(govTalkErrors);

            final Element error = new Element("Error", GOVTALK_ENVELOPE_NS);
            govTalkErrors.addContent(error);

            final Element raisedBy = new Element("RaisedBy", GOVTALK_ENVELOPE_NS);

            raisedBy.addContent(params.getGovTalkRaisedBy());
            error.addContent(raisedBy);

            final Element number = new Element("Number", GOVTALK_ENVELOPE_NS);
            number.addContent("3001");
            error.addContent(number);

            final Element type = new Element("Type", GOVTALK_ENVELOPE_NS);
            type.addContent("business");
            error.addContent(type);

            final Element text = new Element("Text", GOVTALK_ENVELOPE_NS);
            text.addContent(params.getFailureMessage());
            error.addContent(text);
        }
        return result;
    }

    private static Element generateBody(final Element e) {
        final Element body = new Element("Body", GOVTALK_ENVELOPE_NS);
        if (e != null) {
            e.addContent(body);
        }
        return body;
    }

    /**
     * writes the header XML to the element
     * 
     * @element the element to write to
     * 
     */
    public static void writeHeader(final Element element, final int errorCount, final GovTalkHeaderDetails govTalkHeader,
            final EricResponseParams params) {
        final Element header = new Element("Header", GOVTALK_ENVELOPE_NS);
        element.addContent(header);
        {
            final Element messageDetails = new Element("MessageDetails", GOVTALK_ENVELOPE_NS);
            header.addContent(messageDetails);

            // message details
            {
                {
                    final Element messageClass = new Element("Class", GOVTALK_ENVELOPE_NS);

                    String messageClassString = govTalkHeader.getMessageClass();
                    if (params.getOverrideMessageClass() != null) {
                        messageClassString = params.getOverrideMessageClass();
                    }
                    messageClass.addContent(messageClassString);
                    messageDetails.addContent(messageClass);
                }
                {
                    final Element qualifier = new Element("Qualifier", GOVTALK_ENVELOPE_NS);
                    if (errorCount > 0) {
                        qualifier.addContent("error");
                    } else {
                        qualifier.addContent("response");
                    }
                    messageDetails.addContent(qualifier);
                }
                {
                    final Element function = new Element("Function", GOVTALK_ENVELOPE_NS);
                    function.addContent("submit");
                    messageDetails.addContent(function);
                }
                {
                    final Element correlationID = new Element("CorrelationID", GOVTALK_ENVELOPE_NS);
                    correlationID.addContent(govTalkHeader.getCorrelationId());
                    messageDetails.addContent(correlationID);
                }
                {
                    messageDetails.addContent(new Element("ResponseEndPoint", GOVTALK_ENVELOPE_NS));
                }
                {
                    final Element transformation = new Element("Transformation", GOVTALK_ENVELOPE_NS);
                    transformation.addContent("XML");
                    messageDetails.addContent(transformation);
                }
                {
                    final Element gatewayTest = new Element("GatewayTest", GOVTALK_ENVELOPE_NS);
                    gatewayTest.addContent(govTalkHeader.getGatewayTest());
                    messageDetails.addContent(gatewayTest);
                }
                {
                    // We can leave this empty and the gateway will fill it in
                    // on
                    // the way back out
                    final Element gatewayTimestamp = new Element("GatewayTimestamp", GOVTALK_ENVELOPE_NS);
                    messageDetails.addContent(gatewayTimestamp);
                }
            }

            final Element senderDetails = new Element("SenderDetails", GOVTALK_ENVELOPE_NS);
            header.addContent(senderDetails);

            /**
             * Don't populate the sender details back to DIS. - Doug 16/3/2006
             */
            // if (false) { // !!!!! code disabled !!!!!
            // final Element idAuthentication = new Element("IDAuthentication",
            // GOVTALK_ENVELOPE_NS);
            // senderDetails.addContent(idAuthentication);
            // {
            // final Element senderID = new Element("SenderID",
            // GOVTALK_ENVELOPE_NS);
            // if (govTalkHeader != null) {
            // senderID.addContent(govTalkHeader.getSenderId());
            // }
            // idAuthentication.addContent(senderID);
            // }
            // }

            if (govTalkHeader.getEmailAddress() != null && !govTalkHeader.getEmailAddress().trim().equals("")) {
                final Element emailAddress = new Element("EmailAddress", GOVTALK_ENVELOPE_NS);
                emailAddress.addContent(govTalkHeader.getEmailAddress());
                senderDetails.addContent(emailAddress);
            }
        }
    }

}
