/*
 * Created on Nov 7, 2005
 * 
 * $Id: JDOMGeneratorList.java,v 1.31 2006/08/10 15:19:11 dec Exp $
 */
package ericResponse.output;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

import org.jdom.Comment;
import org.jdom.Element;

import ericResponse.output.error.EricResponseError;

/**
 * This list wraps a list of EricResponseErrors and will return JDOM Elements
 * from the get() method which respresent those errors as JDOM trees. The trees
 * will be built on the fly, not stored, so this provides a memory-efficient way
 * of generating the trees for a large number of errors without having to create
 * them all in memory at the same time.
 * 
 * The structure of the XML being generated has an "Application" element before
 * the "Error" elements. That "Application" element holds the error count. For
 * this reason, get(0) from this list will return the tree for the "Application"
 * element with the error count filled in and get(1) and above will return the
 * trees for the actual "Errors". Hence the size of this list will always be 1
 * more than the size of the underlying list.
 * 
 * A number of the List methods are not implemented because they are not needed
 * by the code which is intending to use this. Also, a number of the methods,
 * particularly contains() and related related methods, have trivial
 * implementations as they are not particularly meaningful. The unimilemented
 * methods will throw UnsupportOperationExceptions if called. Particularly note
 * that the iterator() method is not implemented.
 */

public class JDOMGeneratorList implements List<Element> {
    private final List<EricResponseError> wrappedList;

    private String namespaceURL = "";

    /**
     * @directed
     */
    private final EricResponseParams params;

    /**
     * @param ericResponseErrors
     *            the list of EricResponseErrors which will be converted to JDOM
     *            Element trees by the get() method.
     */
    public JDOMGeneratorList(final List<EricResponseError> ericResponseErrors, final EricResponseParams params) {
        this.wrappedList = ericResponseErrors;
        this.params = params;
    }

    public JDOMGeneratorList(final List<EricResponseError> ericResponseErrors, final String namespaceURL,
            final EricResponseParams params) {
        this.wrappedList = ericResponseErrors;
        this.namespaceURL = namespaceURL;
        this.params = params;
    }

    /**
     * The size will be 1 greater than the size of the wrapped list of
     * EricResponseErrors. This is because the first element in the list (i.e.
     * get(0)) will return an "Application" element containing the number of
     * errors. get(1) and above will return the "Error" elements themselves.
     */
    public int size() {
        // Add one to account for the Application element which comes out first.
        return wrappedList.size() + 1;
    }

    /**
     * get(0) will always return the "Application" element tree which contains
     * the error count. get(1+) will return the "Error" element trees for the
     * errors in the underlying list.
     * 
     * @return a JDOM Element object.
     */
    public Element get(final int i) {
        if (i == 0) {
            final Element element = new Element("MessageCount", namespaceURL);
            int size = wrappedList.size();
            if (params.getMaxErrorsToReport() > 0) {
                if (size > params.getMaxErrorsToReport()) {
                    size = params.getMaxErrorsToReport();
                }
            }

            element.setText(Integer.toString(size));
            return element;
        } else {
            final EricResponseError ere = wrappedList.get(i - 1);
            return buildJDOMTree(ere);
        }
    }

    private Element buildJDOMTree(final EricResponseError ere) {
        final Element errorElement = new Element("Error", namespaceURL);

        if (params.getOutputExtraDetails()) {
            final List<String> messages = ere.generateExtraDetailMessages();
            if (messages != null) {
                for (final String element : messages) {
                    /*
                     * Need to deal with any occurrences of '--' in the comment
                     * string. JDOM won't allow these as they'll get confused
                     * with the end-of-comment marker. Here we backslash them to
                     * keep JDOM happy. The first replace() will deal with an
                     * even number of hyphens. The second replace deals with the
                     * last hyphen pair when there are an odd number.
                     */
                    errorElement.addContent(new Comment(element.replace("--", "\\-\\-").replace("--", "-\\-")));
                }
            }
        }

        addRaisedBy(errorElement);
        addNumber(ere, errorElement);
        addType(ere, errorElement);
        addText(ere, errorElement);
        addLocation(ere, errorElement);

        return errorElement;
    }

    private void addLocation(final EricResponseError ere, final Element errorElement) {
        final Element location = new Element("Location", namespaceURL);
        final String locationString = ere.getLocator();

        location.addContent(locationString);
        errorElement.addContent(location);
    }

    private void addText(final EricResponseError ere, final Element errorElement) {
        final Element text = new Element("Text", namespaceURL);
        text.addContent(ere.getMessageText());
        errorElement.addContent(text);
    }

    private void addType(final EricResponseError ere, final Element errorElement) {
        final Element type = new Element("Type", namespaceURL);
        type.addContent(ere.getErrorType().toString());
        errorElement.addContent(type);
    }

    private void addNumber(final EricResponseError ere, final Element errorElement) {
        final Element number = new Element("Number", namespaceURL);
        number.addContent(ere.getCode());
        errorElement.addContent(number);
    }

    private void addRaisedBy(final Element errorElement) {
        final Element raisedBy = new Element("RaisedBy", namespaceURL);
        raisedBy.addContent(getRaisedBy());
        errorElement.addContent(raisedBy);
    }

    public String getRaisedBy() {
        return params.getRaisedBy();
    }

    // =======================================
    // Trivial implementations
    // =======================================

    public void clear() {
        wrappedList.clear();
    }

    public boolean isEmpty() {
        return wrappedList.isEmpty();
    }

    public Object[] toArray() {
        return wrappedList.toArray();
    }

    public int indexOf(final Object arg0) {
        // Don't support looking up objects. Doesn't really make sense
        return -1;
    }

    public int lastIndexOf(final Object arg0) {
        return -1;
    }

    public boolean contains(final Object arg0) {
        return false;
    }

    public boolean remove(final Object arg0) {
        return false;
    }

    public boolean containsAll(final Collection<?> arg0) {
        return false;
    }

    public boolean removeAll(final Collection<?> arg0) {
        return false;

    }

    public boolean retainAll(final Collection<?> arg0) {
        return false;
    }

    // ===============================================
    // Unimplemented List methods below here
    // ===============================================

    public Element remove(final int i) {
        throw new UnsupportedOperationException("Read-only list. Cannot remove items.");
    }

    public void add(final int arg0, final Element arg1) {
        throw new UnsupportedOperationException("Read-only list. Cannot add items.");
    }

    public Iterator<Element> iterator() {
        throw new UnsupportedOperationException("implement me!");
    }

    public List<Element> subList(final int arg0, final int arg1) {
        throw new UnsupportedOperationException("implement me!");
    }

    public ListIterator<Element> listIterator() {
        throw new UnsupportedOperationException("implement me!");
    }

    public ListIterator<Element> listIterator(final int arg0) {
        throw new UnsupportedOperationException("implement me!");
    }

    public Element set(final int arg0, final Element arg1) {
        throw new UnsupportedOperationException("Read-only list. Cannot set items.");
    }

    public boolean addAll(final int arg0, final Collection<? extends Element> arg1) {
        throw new UnsupportedOperationException("Read-only list. Cannot add items.");
    }

    public boolean addAll(final Collection<? extends Element> arg0) {
        throw new UnsupportedOperationException("Read-only list. Cannot add items.");
    }

    public boolean add(final Element arg0) {
        throw new UnsupportedOperationException("Read-only list. Cannot add items.");
    }

    public <T> T[] toArray(final T[] arg0) {
        throw new UnsupportedOperationException("implement me!");
    }

}
