/*
 * Created on Nov 3, 2005
 * 
 * $Id: EricResponseErrorBean.java,v 1.18 2006/08/10 15:19:11 dec Exp $
 */
package ericResponse.output.error;

import hmrc.bizrules.QualityStage;

import java.util.ArrayList;
import java.util.List;

import processingError.ErrorProcessor;
import processingError.ProcessingError;
import bizrules.errors.MonetaryDiscrepancy;
import ericResponse.output.translator.TranslationContext;
import ericResponse.output.translator.mapper.ErrorMapper;

/**
 * Simple bean to represent an error which will be sent in the ERIC XML
 * Response.
 */
public class EricResponseErrorBean implements EricResponseError {

    protected ProcessingError underlyingError;

    protected ERErrorDetail detail;

    protected Object extraInfo;

    protected TranslationContext translationContext;

    protected ErrorMapper mapper;

    public EricResponseErrorBean(final ERErrorDetail detail, final ProcessingError underlyingError) {
        this(detail, underlyingError, null, null);
    }

    public EricResponseErrorBean(final ERErrorDetail detail, final ProcessingError underlyingError,
            final TranslationContext translationContext) {
        this(detail, underlyingError, translationContext, null);
    }

    /**
     * @param detail
     *            code, locator, error message, etc about the error (some
     *            details specific to ERIC)
     * @param underlyingError
     *            the processing error which generated this error. null is ok.
     * @param mapper
     *            the ErrorMapper instance that greating this error, or null if
     *            no mapper was involved (e.g. it was a generic error).
     */
    public EricResponseErrorBean(final ERErrorDetail detail, final ProcessingError underlyingError,
            final TranslationContext translationContext, final ErrorMapper mapper) {
        this.detail = detail;
        this.underlyingError = underlyingError;
        this.translationContext = translationContext;
        this.mapper = mapper;

        // get extra info, if any
        if (underlyingError != null) {
            setExtraInfo(underlyingError.getExtraInfo());
        }
    }

    public String getCode() {
        return detail.getCode();
    }

    public String getLocator() {
        String locatorText = detail.getLocator();
        if (getExtraInfo() != null && !getExtraInfo().toString().equals("")) {
            locatorText += ":" + getExtraInfo();
        }
        return locatorText;
    }

    public String getMessageText() {
        String result = detail.getMessageText();
        if (underlyingError != null) {
            result = replaceVariables(detail.getMessageText(), underlyingError.getValueInError());
        }
        return result;
    }

    @Override
    public boolean equals(final Object other) {
        boolean result = false;

        if (other != null && other.getClass() == this.getClass()) {
            final EricResponseErrorBean that = (EricResponseErrorBean) other;

            result = this.getCode() == that.getCode() && this.getLocator().equals(that.getLocator())
                    && this.getMessageText().equals(that.getMessageText());
        }
        return result;
    }

    @Override
    public int hashCode() {
        return (getCode() + getLocator() + getMessageText()).hashCode();
    }

    @Override
    public String toString() {
        return "EricReponseError [BRCode =  '" + getBRCode() + "', mappingCode = '" + "', element = '" + "', locator = '"
                + getLocator() + "', code = " + getCode() + ", message text = '" + getMessageText() + "]";
    }

    public Object getExtraInfo() {
        return extraInfo;
    }

    public void setExtraInfo(final Object extraInfo) {
        this.extraInfo = extraInfo;
    }

    public EricErrorType getErrorType() {
        return detail.getErrorType();
    }

    private static String replaceVariables(final String messageText, final Object valueInError) {
        final String originalText = messageText;
        String updatedText = originalText;

        if (valueInError == null) {
            return originalText;
        }

        if (valueInError instanceof MonetaryDiscrepancy) {
            final MonetaryDiscrepancy m = (MonetaryDiscrepancy) valueInError;
            updatedText = originalText.replaceAll("<supplied_value>", m.getSuppliedValue().toString());
            updatedText = updatedText.replaceAll("<calculated_value>", m.getCalculatedValue().toString());
        } else {
            updatedText = originalText.replaceAll("<value>", valueInError.toString());
        }
        return updatedText;
    }

    /**
     * @return the ProcessingError which mapped to this EricResponseError.
     */
    public Object getUnderlyingError() {
        return underlyingError;
    }

    public String getBRCode() {
        return detail.getBRCode();
    }

    public QualityStage getStage() {
        return detail.getStage();
    }

    public ErrorProcessor getErrorProcessor() {
        return detail.getErrorProcessor();
    }

    public List<String> generateExtraDetailMessages() {
        final List<String> messages = new ArrayList<String>();

        if (getUnderlyingError() != null) {
            messages.add(" underlying error: " + getUnderlyingError());
        }

        messages.add(" error type: '" + getErrorType() + "'. ");
        messages.add(" business rule code: '" + getBRCode() + "'. ");

        if (getUnderlyingError() instanceof ProcessingError) {
            final Object value = ((ProcessingError) getUnderlyingError()).getValueInError();
            if (value instanceof MonetaryDiscrepancy) {
                final MonetaryDiscrepancy monetaryDiscrepancy = (MonetaryDiscrepancy) value;
                messages.add(" monetary discrepancy: '" + monetaryDiscrepancy.getDiscrepancy() + "'. ");
                messages.add(" monetary calculated value: '" + monetaryDiscrepancy.getCalculatedValue() + "'. ");
                messages.add(" monetary supplied value: '" + monetaryDiscrepancy.getSuppliedValue() + "'. ");
            }
        }

        messages.add(" quality stage: " + getStage() + ".");

        return messages;
    }
}
