/*
 * Created on Nov 3, 2005
 * 
 * $Id: CompositeErrorTranslator.java,v 1.9 2006/08/10 15:19:11 dec Exp $
 */
package ericResponse.output.translator;

import java.util.ArrayList;
import java.util.List;

import processingError.ProcessingError;
import ericResponse.output.EricResponseParams;
import ericResponse.output.error.EricResponseError;
import ericResponse.output.error.EricResponseErrorFactory;
import ericResponse.output.translator.genericerror.GenericErrorMapper;
import ericResponse.output.translator.genericerror.RegistryBasedGenericErrorTranslator;
import ericResponse.output.translator.genericerror.SingleGenericErrorTranslator;
import ericResponse.output.translator.mapper.ErrorMapper;
import ericResponse.output.translator.mapper.RegistryBasedErrorTranslator;
import ericResponse.output.translator.mapper.SingleMapperErrorTranslator;

/**
 * This composite class provides a way of implementing one or more actual error
 * translators. A given ProcessingError will be passed to each translator until
 * one of them returns a non-null EricResponseError.
 * 
 * @author Doug Clinton
 * 
 */
public class CompositeErrorTranslator implements EricErrorTranslator {

    private final List<EricErrorTranslator> mapperTranslatorList = new ArrayList<EricErrorTranslator>();

    private final List<EricErrorTranslator> genericTranslatorList = new ArrayList<EricErrorTranslator>();

    private CatchAllTranslator catchAllTranslator = null;

    private final EricResponseParams params;

    public CompositeErrorTranslator(final EricResponseParams params) {
        this(params, true);
    }

    public CompositeErrorTranslator(final EricResponseParams params, final boolean enableCatchAll) {
        this.params = params;
        if (enableCatchAll)
            catchAllTranslator = new CatchAllTranslator();
    }

    public void addErrorMapper(final ErrorMapper mapper, final EricResponseErrorFactory ereFactory) {
        mapperTranslatorList.add(new SingleMapperErrorTranslator(mapper, ereFactory));
    }

    public void addErrorMappers(final String basePackageName, final EricResponseErrorFactory ereFactory) {
        mapperTranslatorList.add(new RegistryBasedErrorTranslator(params, basePackageName, ereFactory));
    }

    public void addGenericMapper(final GenericErrorMapper mapper) {
        genericTranslatorList.add(new SingleGenericErrorTranslator(mapper));
    }

    public void addGenericMappers(final String basePackageName) {
        genericTranslatorList.add(new RegistryBasedGenericErrorTranslator(params, basePackageName));
    }

    /**
     * Translates the processing error to an eric response error according to
     * the rules for the given tax year.
     * 
     * @return the EricResponseError corresponding to the ProcessingError. This
     *         will never be null
     */
    public EricResponseError translateError(final ProcessingError pe, final TranslationContext context) {
        EricResponseError result = null;

        result = translateError(mapperTranslatorList, pe, context);
        if (result != null)
            return result;

        result = translateError(genericTranslatorList, pe, context);
        if (result != null)
            return result;

        if (catchAllTranslator != null) {
            result = catchAllTranslator.translateError(pe, context);
        }

        return result;
    }

    private EricResponseError translateError(final List<EricErrorTranslator> translatorList, final ProcessingError pe,
            final TranslationContext context) {
        for (int i = 0; i < translatorList.size(); i++) {
            final EricErrorTranslator translator = translatorList.get(i);
            final EricResponseError result = translator.translateError(pe, context);
            if (result != null)
                return result;
        }
        return null;
    }
}
