package ericResponse.output.translator.mapper;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import processingError.ProcessingError;

import com.gsl.util.classlookup.CachingClassCollector;
import com.gsl.util.classlookup.ClassNameCollector;
import com.gsl.util.classlookup.CompositeInterfaceCollector;
import com.gsl.util.classlookup.InterfaceInstanceCollector;

import ericResponse.output.EricResponseParams;
import ericResponse.output.error.EricResponseError;
import ericResponse.output.error.EricResponseErrorFactory;
import ericResponse.output.translator.TranslationContext;

public class RegistryBasedErrorTranslator extends AbstractMapperErrorTranslator {

    private final Map<String, Set<ErrorMapper>> registry = new HashMap<String, Set<ErrorMapper>>();

    /**
     * This constructor will use reflection to scan the base package and its
     * sub-packages to find classes which implement the ErrorMapper interface
     * and register them.
     * 
     * @param basePackageName
     */
    @SuppressWarnings("unchecked")
    public RegistryBasedErrorTranslator(final EricResponseParams params, final String basePackageName,
            final EricResponseErrorFactory ereFactory) {
        super(ereFactory);

        final Class<ErrorMapper> interfaceToCollect = ErrorMapper.class;
        ClassNameCollector baseCollector;

        if (params.getForceDynamicClassCollection()) {
            baseCollector = new InterfaceInstanceCollector(basePackageName, interfaceToCollect);
        } else {
            baseCollector = new CompositeInterfaceCollector(basePackageName, interfaceToCollect);
        }

        final ClassNameCollector collector = new CachingClassCollector(baseCollector);
        final Collection<String> classNames = collector.collectClassNames();

        if (classNames.size() == 0) {
            getLogger().severe(
                    "No error mappers were loaded from " + basePackageName
                            + ". This probably indicates a problem with the classpath.");
        }

        for (final Object element : classNames) {
            final String mapperName = (String) element;
            Class<ErrorMapper> mapperClass;
            try {
                mapperClass = (Class<ErrorMapper>) Class.forName(mapperName);
                final ErrorMapper mapper = mapperClass.newInstance();
                registerErrorMapper(mapper);
            } catch (final ClassNotFoundException e) {
                getLogger().warning("Could not find class " + mapperName);
            } catch (final InstantiationException e) {
                getLogger().warning("Could not instantiate mapper " + mapperName);
            } catch (final IllegalAccessException e) {
                getLogger().warning("Could not access mapper class " + mapperName);
            }
        }
    }

    public void registerErrorMapper(final ErrorMapper mapper) {
        final String[] ruleNames = mapper.getRuleNames();
        for (final String element : ruleNames) {
            registerErrorMapper(element, mapper);
        }
    }

    private void registerErrorMapper(final String ruleName, final ErrorMapper mapper) {
        Set<ErrorMapper> mappers = getMappersForRuleName(ruleName);
        if (mappers == null) {
            mappers = new HashSet<ErrorMapper>();
            registry.put(ruleName, mappers);
        }
        mappers.add(mapper);
    }

    public EricResponseError translateError(final ProcessingError pe, final TranslationContext context) {
        EricResponseError response = null;

        final Set<ErrorMapper> mappers = getMappersForRuleName(pe.getRuleName());
        if (mappers != null) {
            for (final ErrorMapper mapper : mappers) {
                response = mapError(mapper, pe, context);
                if (response != null) {
                    break;
                }
            }
        }

        return response;
    }

    /**
     * This method is exposed to support auditing and reporting.
     */
    public Set<ErrorMapper> getMappersForRuleName(final String ruleName) {
        return registry.get(ruleName);
    }
}
