package hmrc.validator;

import govtalk.header.GovTalkHeaderDetails;
import hmrc.bizrules.QualityStage;
import hmrc.validator.result.ValidationStatus;

import java.io.Writer;
import java.util.Collections;
import java.util.List;

import processingError.ProcessingError;

import com.gsl.docValidator.DocumentValidatorImplementor;
import com.gsl.docValidator.ValidationParams;
import com.gsl.docValidator.ValidationResultInterface;
import com.gsl.docValidator.ValidatorProvider;

import ericResponse.output.*;
import ericResponse.output.error.EricResponseError;
import ericResponse.output.error.EricResponseErrorFactory;
import ericResponse.output.translator.TranslationContext;

/**
 * This class provides a base implementation of an ERIC response post validation
 * delegate suitable for subclassing by HMRC validator implementations.
 * <p>
 * It implements {@link #createResult(List, DocumentValidatorImplementor)}
 * itself. Subclasses should generally just override the protected abstract
 * {@link #createHMRCResult()} method to instantiate and return the appropriate
 * {@link HMRCValidationResultBean} subclass.
 * <p>
 * Note that the {@link EricResponsePostValidationDelegate} superclass defines
 * the {@link #createTranslationContext()} and
 * {@link #createEricResponseErrorFactory()} methods that subclasses of this
 * class may also consider overriding.
 * <p>
 * Note also that it is the {@link ValidatorProvider} implementation's
 * responsibility to ensure that the
 * {@link #setGovTalkHeaderDetails(GovTalkHeaderDetails)} and
 * {@link #setIREnvelopeDetails(IREnvelopeDetails)} methods are called, passing
 * the relevant detail objects. The former is taken care of by the
 * {@link GatewayEnvelopeValidatorProvider} provider implementation. The latter
 * will generally be up to the particular validator implementation to take of.
 * <p>
 * For more advanced requirements, the implementor may use this class in a
 * slightly different, more flexible way. A provider may call
 * {@link #setDelegateHandler(DelegateHandler)} on this class to set a
 * DelegateHandler instance that will take over certain aspects of the
 * delegate's responsibilities.
 * <p>
 * If a delegate handler is set (via the
 * {@link #setDelegateHandler(DelegateHandler) method, then it will be
 * responsible for implementing the methods mentioned above, such as
 * {@link #createEricResponseErrorFactory()}. Also, in this case it is not
 * necessary to override {@link #createHMRCResult()}, since the delegate will
 * be responsible for this via its own API.
 * <p>
 * This approach provides more flexibility when implementing a validator. A
 * typical use case is when there are multiple providers for multiple expected
 * document types, each of which requires different post validation processing.
 * 
 * @see DelegateHandler
 * 
 * @author jesse
 */
public abstract class HMRCValidationDelegate extends EricResponsePostValidationDelegate {

    private DelegateHandler delegateHandler;

    private GovTalkHeaderDetails govTalkHeaderDetails;

    private IREnvelopeDetails irEnvelopeDetails;

    public HMRCValidationDelegate(final EricResponseParams responseParams, final Writer output) {
        super(responseParams, output);
    }

    public void setDelegateHandler(final DelegateHandler delegateHandler) {
        this.delegateHandler = delegateHandler;
    }

    public void setGovTalkHeaderDetails(final GovTalkHeaderDetails govTalkHeaderDetails) {
        this.govTalkHeaderDetails = govTalkHeaderDetails;
    }

    public GovTalkHeaderDetails getGovTalkHeaderDetails() {
        return govTalkHeaderDetails;
    }

    public void setIREnvelopeDetails(final IREnvelopeDetails irEnvelopeDetails) {
        this.irEnvelopeDetails = irEnvelopeDetails;
    }

    public IREnvelopeDetails getIREnvelopeDetails() {
        return irEnvelopeDetails;
    }

    public ValidationResultInterface createResult(final List<ProcessingError> processingErrors, final ValidationParams params) {
        final EricErrorResponseGenerator ericErrorResponseGenerator = new EricErrorResponseGenerator();
        addResponseGenerator(ericErrorResponseGenerator);

        final HMRCValidationResultInterface result = getDelegateHandler().createResult(this, params);

        final List<EricResponseError> ericErrors = mapErrors(processingErrors);
        result.setEricResponseErrors(ericErrors);

        countErrors(result, ericErrors);

        result.setGovTalkHeaderDetails(govTalkHeaderDetails);
        result.setIrEnvelopeDetails(irEnvelopeDetails);

        getDelegateHandler().processResult(result, params);
        ericErrorResponseGenerator.addApplicationElementGenerators(makeApplicationElementGenerators(result, params));

        generateResponse(ericErrors);

        return result;
    }

    protected List<ApplicationElementGenerator> makeApplicationElementGenerators(final HMRCValidationResultInterface result,
            final ValidationParams params) {
        return Collections.emptyList();
    }

    protected DelegateHandler getDelegateHandler() {
        if (delegateHandler == null) {
            delegateHandler = createDefaultDelegateHandler();
        }
        return delegateHandler;
    }

    @Override
    protected EricResponseErrorFactory createEricResponseErrorFactory() {
        return getDelegateHandler().createEricResponseErrorFactory();
    }

    @Override
    protected TranslationContext createTranslationContext() {
        return getDelegateHandler().createTranslationContext();
    }

    protected HMRCValidationResultInterface createHMRCResult() {
        throw new UnsupportedOperationException();
    }

    protected DelegateHandler createDefaultDelegateHandler() {
        return new DefaultDelegateHandler();
    }

    private static void countErrors(final HMRCValidationResultInterface result, final List<EricResponseError> ericErrors) {
        int qs2Cnt = 0, qs3Cnt = 0;

        for (final EricResponseError error : ericErrors) {
            if (error.getStage().equals(QualityStage.QS2)) {
                qs2Cnt++;
            } else if (error.getStage().equals(QualityStage.QS3)) {
                qs3Cnt++;
            }
        }

        result.getQS2Result().setErrorCount(qs2Cnt);
        result.getQS3Result().setErrorCount(qs3Cnt);

        result.getQS2Result().setValidationStatus(qs2Cnt > 0 ? ValidationStatus.FAILED : ValidationStatus.PASSED);
        result.getQS3Result().setValidationStatus(qs3Cnt > 0 ? ValidationStatus.FAILED : ValidationStatus.PASSED);
    }

    protected static class DefaultDelegateHandler implements DelegateHandler {

        public HMRCValidationResultInterface createResult(final HMRCValidationDelegate delegate, final ValidationParams params) {
            return delegate.createHMRCResult();
        }

        public void processResult(final HMRCValidationResultInterface result, final ValidationParams params) {
        }

        public EricResponseErrorFactory createEricResponseErrorFactory() {
            return null;
        }

        public TranslationContext createTranslationContext() {
            return null;
        }
    }
}
