package hmrc.validator;

import java.util.*;

import com.gsl.docValidator.ValidationParams;

import ericResponse.output.EricResponseParams;
import govtalk.header.GatewayEnvelopeVersion;

/**
 * Base class for providing parameters for HMRC validator implementations.
 * Extends the base validation framework parameters class and adds some
 * parameters specific to all HMRC validator implementations. Concrete HMRC
 * validator implementations should provide a parameters class that extends this
 * class.
 */
public abstract class HMRCValidationParams extends ValidationParams {

    private EricResponseParams responseParams = new EricResponseParams();

    private GatewayEnvelopeVersion envelopeVersion = GatewayEnvelopeVersion.V2;

    private String ericEnvelopeVersion = "1.5";

    private boolean suppressGatewayEnvelopeErrors = true;

    private Map<String, Set<String>> hmrcRulePackages = null;

    @Override
    public String toString() {
        String s = "HMRCValidationParams [";
        s += super.toString();
        s += ",\n";
        s += "envelopeVersion=" + envelopeVersion;
        s += ", ";
        s += "suppressGatewayEnvelopeErrors=" + suppressGatewayEnvelopeErrors;
        s += ", ";
        s += "hmrcRulePackages=" + hmrcRulePackages;
        s += ",\n";
        s += "responseParams=" + responseParams;
        s += "]";
        return s;
    }

    /**
     * Gets the ERIC response parameters associated with this parameters
     * instance.
     * 
     * @return The ERIC response parameters associated with this parameters
     *         instance.
     */
    public EricResponseParams getResponseParams() {
        return responseParams;
    }

    /**
     * Sets the ERIC response parameters associated with this parameters
     * instance.
     * <p>
     * Should not be null.
     * 
     * @return The ERIC response parameters to associate with this parameters
     *         instance.
     */
    public void setResponseParams(final EricResponseParams errorParams) {
        this.responseParams = errorParams;
    }

    /**
     * Gets the gateway envelope version to expect for the document being
     * validated.
     * 
     * @see GatewayEnvelopeVersion
     * 
     * @return The gateway envelope version to expect for the document being
     *         validated.
     * @deprecated
     */
    @Deprecated
    public GatewayEnvelopeVersion getEnvelopeVersion() {
        return envelopeVersion;
    }

    /**
     * Sets the gateway envelope version for the document being validated.
     * 
     * @see GatewayEnvelopeVersion
     * 
     * @param envelopeVersion
     *            The gateway envelope version for the document being validated.
     * 
     * @deprecated We no longer choosed the schema based on the envelope version -
     *             we always validate with v2.0. You no longer need to call this
     *             method and it will be removed in a future version of this
     *             class.
     */
    @Deprecated
    public void setEnvelopeVersion(final GatewayEnvelopeVersion envelopeVersion) {
        this.envelopeVersion = envelopeVersion;
    }

    public void setEricEnvelopeVersion(final String ericEnvelopeVersion) {
        this.ericEnvelopeVersion = ericEnvelopeVersion;
    }

    public String getEricEnvelopeVersion() {
        return ericEnvelopeVersion;
    }

    /**
     * Indicates whether or not to suppress errors found in the gateway envelope
     * of the document being validated. These are errors are normally suppressed
     * since we normally don't want envelope errors alone to prevent submissions
     * being successfully passed through the system.
     * <p>
     * Defaults to true.
     * 
     * @return true if gateway envelope errors should be suppressed, false if
     *         not.
     */
    public boolean isSuppressGatewayEnvelopeErrors() {
        return suppressGatewayEnvelopeErrors;
    }

    /**
     * Set this parameter to true to suppress errors found in the gateway
     * envelope of the document being validated.
     * 
     * @see #isSuppressGatewayEnvelopeErrors()
     * 
     * @param suppressGatewayEnvelopeErrors
     *            true to suppress gateway envelope errors in the document being
     *            validated, false to include them.
     */
    public void setSuppressGatewayEnvelopeErrors(final boolean suppressGatewayEnvelopeErrors) {
        this.suppressGatewayEnvelopeErrors = suppressGatewayEnvelopeErrors;
    }

    public Set<String> getHmrcRulePackages(final String namespace) {
        if (hmrcRulePackages == null) {
            return Collections.emptySet();
        }
        Set<String> rulePackages = hmrcRulePackages.get(namespace);
        if (rulePackages == null) {
            rulePackages = Collections.emptySet();
        }
        return rulePackages;
    }

    public void setHmrcRulePackages(final Map<String, Set<String>> hmrcRulePackages) {
        this.hmrcRulePackages = hmrcRulePackages;
    }

    public void setHmrcRulePackages(final String namespace, final Set<String> rulePackages) {
        if (hmrcRulePackages == null) {
            hmrcRulePackages = new HashMap<String, Set<String>>();
        }
        hmrcRulePackages.put(namespace, rulePackages);
    }

    public void addHmrcRulePackage(final String namespace, final String rulePackage) {
        if (hmrcRulePackages == null) {
            hmrcRulePackages = new HashMap<String, Set<String>>();
        }
        Set<String> rulePackages = hmrcRulePackages.get(namespace);
        if (rulePackages == null) {
            rulePackages = new HashSet<String>();
            rulePackages.add(rulePackage);
        }
        hmrcRulePackages.put(namespace, rulePackages);
    }
}
