package hmrc.validator;

import govtalk.header.GovTalkHeaderDetails;
import hmrc.bizrules.HMRCBusinessRuleError;
import hmrc.bizrules.QualityStage;
import hmrc.validator.result.HMRCValidationResult;
import hmrc.validator.result.QSResult;

import java.math.BigDecimal;
import java.util.List;

import bizrules.errors.MonetaryDiscrepancy;

import com.gsl.docValidator.ValidationResultBean;

import ericResponse.output.IREnvelopeDetails;
import ericResponse.output.error.EricResponseError;

/**
 * Base class for returning HMRC validation results from a HMRC validator back
 * to the client.
 * <p>
 * This class should be extended by HMRC validator implementations wishing to
 * cater for document-type specific validation result information.
 * 
 * @see HMRCValidationResult
 */
public abstract class HMRCValidationResultBean extends ValidationResultBean implements HMRCValidationResultInterface {

    private transient List<EricResponseError> ericResponseErrors;

    private transient GovTalkHeaderDetails govTalkHeaderDetails;

    private transient IREnvelopeDetails irEnvelopeDetails;

    private final QSResult QS3Result = new QSResult();

    private final QSResult QS2Result = new QSResult();

    public QSResult getQS2Result() {
        return QS2Result;
    }

    public QSResult getQS3Result() {
        return QS3Result;
    }

    public BigDecimal getQS3MonetaryDiscrepancy() {
        BigDecimal totalDiscrepancy = new BigDecimal("0.00");
        for (EricResponseError error : ericResponseErrors) {
            if (error.getUnderlyingError() instanceof HMRCBusinessRuleError) {
                HMRCBusinessRuleError bre = (HMRCBusinessRuleError) error.getUnderlyingError();
                if (bre.getQualityStage() == QualityStage.QS3) {
                    if (bre.getValueInError() != null && bre.getValueInError() instanceof MonetaryDiscrepancy) {
                        MonetaryDiscrepancy md = (MonetaryDiscrepancy) bre.getValueInError();
                        totalDiscrepancy = totalDiscrepancy.add(md.getDiscrepancy());
                    }
                }
            }
        }
        return totalDiscrepancy;
    }

    public List<EricResponseError> getEricResponseErrors() {
        return ericResponseErrors;
    }

    public void setEricResponseErrors(final List<EricResponseError> ericResponseErrors) {
        this.ericResponseErrors = ericResponseErrors;
    }

    public GovTalkHeaderDetails getGovTalkHeaderDetails() {
        return govTalkHeaderDetails;
    }

    public void setGovTalkHeaderDetails(final GovTalkHeaderDetails govTalkHeaderDetails) {
        this.govTalkHeaderDetails = govTalkHeaderDetails;
    }

    public IREnvelopeDetails getIrEnvelopeDetails() {
        return irEnvelopeDetails;
    }

    public void setIrEnvelopeDetails(final IREnvelopeDetails irEnvelopeDetails) {
        this.irEnvelopeDetails = irEnvelopeDetails;
    }

    @Override
    public String toString() {
        return "ValidationResult [qs2Result='" + getQS2Result() + ", QS3Result='" + getQS3Result() + "', processing time="
                + getProcessingTime() + "]";
    }
}
