/**
 * 
 */
package hmrc.validator.rules;

import java.util.ArrayList;
import java.util.List;

import xmldoc.DocumentEventType;
import xmldoc.ElementReference;
import xmldoc.SimpleElementReference;
import bizrules.AbstractBusinessRule;
import bizrules.binding.DocumentEventBinding;
import bizrules.registry.DocumentEventDetail;
import bizrules.registry.StartElementDetail;

/**
 * This rule provides some core checks on the presence and position of the
 * IRenvelope element in the submission. The element must have a namespace
 * corresponding to the tax year being checked and the xpath of the parent
 * element should match the parent xpath we are given at construction.
 * 
 * @author Doug Clinton
 */
public class IREnvelopeRule extends AbstractBusinessRule {

    public static final String NO_IR_ENVELOPE = "No IR Envelope";

    private static final String DEFAULT_PARENT_XPATH = "/GovTalkMessage/Body";

    private final String[] namespaces;

    private final List<String> parentXPaths = new ArrayList<String>();

    private boolean irEnvelopeDetected = false;

    private final String[] subcodes = { NO_IR_ENVELOPE };

    @Override
    public String[] getSubcodes() {
        return subcodes;
    }

    public IREnvelopeRule(final String namespace) {
        this(new String[] { namespace }, DEFAULT_PARENT_XPATH);
    }

    public IREnvelopeRule(final String namespace, final String parentXPath) {
        this(new String[] { namespace }, parentXPath);
    }

    public IREnvelopeRule(final String[] namespaces) {
        this(namespaces, DEFAULT_PARENT_XPATH);
    }

    public IREnvelopeRule(final String[] namespaces, final String parentXPath) {
        this.namespaces = namespaces;
        this.parentXPaths.add(parentXPath);
    }

    public IREnvelopeRule(final String[] namespaces, final String[] parentXPaths) {
        this.namespaces = namespaces;
        for (final String parentXPath : parentXPaths) {
            this.parentXPaths.add(parentXPath);
        }
    }

    /**
     * We don't implement processRule(). All our activity is done in the event
     * callbacks. We shouldn't even be a BusinessRule except that we can't
     * generate a BusinessRuleError unless we are one. This shows up a slight
     * problem in the type hierachy which we should address at some point.
     */
    @Override
    protected String[] getInvocationPoints() {
        return new String[] {};
    }

    @Override
    protected String[] getApplicableNamespaces() {
        return new String[] {};
    }

    public void processRule() {
        // do nothing
    }

    @Override
    public boolean isActive() {
        return false;
    }

    public void startElement(final DocumentEventDetail detail) {
        final StartElementDetail startDetail = (StartElementDetail) detail;

        final ElementReference parentElementReference = startDetail.getParentElementReference();

        /*
         * Check that our parent element was as expected - i.e. that we are in
         * the right place in the document.
         */
        if (parentElementReference != null) {
            for (final String parentXPath : parentXPaths) {
                if (parentElementReference.getXPath().equals(parentXPath)) {
                    irEnvelopeDetected = true;
                }
            }
        }
    }

    public void endDocument(final DocumentEventDetail detail) {
        if (!irEnvelopeDetected) {
            raiseError(NO_IR_ENVELOPE);
        }
    }

    @Override
    public List<DocumentEventBinding> getEventBindings() {
        final List<DocumentEventBinding> bindings = new ArrayList<DocumentEventBinding>();
        for (final String element : namespaces) {
            final ElementReference irEnvelopeRef = SimpleElementReference.makeReference(element, "/IRenvelope");
            bindings.add(new DocumentEventBinding(this, irEnvelopeRef, "startElement", DocumentEventType.startElement));
        }
        bindings.add(new DocumentEventBinding(this, null, "endDocument", DocumentEventType.endDocument));
        return bindings;
    }

    @Override
    protected void initializeValueBindings() {
    }

    @Override
    protected void initializeEventBindings() {
    }
}
