/**
 * $Id$
 */
package paye.eoy.bizrules;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import paye.channel.SubmissionChannel;
import paye.eoy.bizrules.helpers.WholePoundsHelper;
import paye.eoy.types.ReturnType;
import paye.eoy.types.SubmissionType;
import paye.multiyear.TaxYear;
import xmldoc.DocumentError;

/**
 * This class provides a common set of features for all rules relating to
 * processing PAYE submissions. In particular, it provides a way for a rule to
 * specify which submssion types, return types and tax years it applies to and
 * uses that information to decide if the rule is active, and also to decide
 * (based on the tax years) which namespaces it applies to.
 * 
 * @author Douglas Clinton
 * @since Jan 9, 2006
 * 
 */
public abstract class AbstractPAYEBusinessRule extends AbstractEOYBusinessRule {

    protected static final Pattern CHARACTER_SET_A = Pattern.compile("[a-zA-Z0-9 .,()/=!\"%&*;<>'+:?-]*");

    private static final SubmissionType[] ALL_SUBMISSION_TYPES = { SubmissionType.COMPLETE, SubmissionType.P14PART,
            SubmissionType.P35PART };

    private static final ReturnType[] ALL_RETURN_TYPES = { ReturnType.ORIGINAL, ReturnType.AMENDED };

    /**
     * Rather than have every rule collect information on submission type,
     * return type and tax year (which would mean a lot of element value
     * callbacks being generated) we focus that information through the
     * SubmissionPropertyCollector. The getProperties() method is thread-safe so
     * what we get here is a thread-global bean which will be fed those details
     * by the XML processor.
     */
    private final PAYESubmissionProperties submissionParameters = PAYESubmissionPropertyCollector.getProperties();

    /**
     * In order to determine if a particular rule is active, we interrogate the
     * subclass to find out which submission types, return types and tax years
     * it applies to and then compare that against the actual submission type,
     * return type and tax year of the submission we are currently processing.
     * If there is a match then the rule is active.
     * <p>
     * This means that it is perfectly safe to simply register all rules for all
     * years at startup as only the ones applicable to the submission being
     * processed will be invoked.
     */
    @Override
    public boolean isActive() {
        final SubmissionType submissionType = submissionParameters.getSubmissionType();
        final ReturnType returnType = submissionParameters.getReturnType();
        final TaxYear taxYear = submissionParameters.getTaxYear();

        return isActiveForSubmissionType(submissionType) && isActiveForReturnType(returnType) && isActiveForTaxYear(taxYear);
    }

    private boolean isActiveForReturnType(final ReturnType returnType) {
        boolean result = false;
        final ReturnType[] returnTypes = getActiveReturnTypes();
        for (final ReturnType element : returnTypes) {
            if (element == returnType) {
                result = true;
                break;
            }
        }
        return result;
    }

    private boolean isActiveForSubmissionType(final SubmissionType submissionType) {
        boolean result = false;
        final SubmissionType[] submissionTypes = getActiveSubmissionTypes();
        for (final SubmissionType element : submissionTypes) {
            if (element == submissionType) {
                result = true;
                break;
            }
        }
        return result;
    }

    /**
     * By default we will be active for all return types. Rules should override
     * this if they wish to restrict the return types they apply to.
     */
    protected ReturnType[] getActiveReturnTypes() {
        final ReturnType[] returnTypes = ALL_RETURN_TYPES;
        return returnTypes;
    }

    /**
     * Also by default we are active for all submission types. Rules should
     * override this if they wish to restrict the list.
     */
    protected SubmissionType[] getActiveSubmissionTypes() {
        final SubmissionType[] activeTypes = ALL_SUBMISSION_TYPES;
        return activeTypes;
    }

    protected SubmissionType getSubmissionType() {
        return submissionParameters.getSubmissionType();
    }

    protected SubmissionChannel getSubmissionChannel() {
        return submissionParameters.getChannel();
    }

    protected TaxYear getTaxYear() {
        return submissionParameters.getTaxYear();
    }

    protected ReturnType getReturnType() {
        return submissionParameters.getReturnType();
    }

    protected void checkCharacterSetA(final String text) {
        final Matcher matcher = CHARACTER_SET_A.matcher(text);
        if (!matcher.matches()) {
            raiseError(DocumentError.FORMAT, text);
        }
    }

    public boolean distinguishWholePoundsError(final DocumentError documentError) {
        if (WholePoundsHelper.checkWholePounds(documentError)) {
            raiseError(WholePoundsHelper.NOT_WHOLE_POUNDS);
            return true;
        } else {
            return false;
        }
    }
}
