/*
 * Created on 30-May-2005
 * 
 * $Id: NumberOfPartsRule.java,v 1.3 2006/05/26 10:45:42 dec Exp $
 *  
 */
package paye.eoy.bizrules;

import paye.channel.SubmissionChannel;
import paye.eoy.types.SubmissionType;
import paye.multiyear.ActiveTaxYear;
import paye.multiyear.TaxYear;
import xmldoc.DocumentError;

/**
 * This rule checks that the NumberOfParts element is present only if the
 * submission type is P35Part and if it is not a P35Part it checks that the
 * NumberOfParts entry is not present.
 * 
 * @author doug, gary, rob
 */
public class NumberOfPartsRule extends AbstractPAYEBusinessRule {

    public static final String NOT_REQUIRED = "not required";

    private String numberOfParts = "";

    private final String[] subcodes = { NOT_REQUIRED, DocumentError.FORMAT, DocumentError.MANDATORY, DocumentError.BLANK,
            DocumentError.MINVALUE, DocumentError.MAXVALUE };

    @Override
    public String[] getSubcodes() {
        return subcodes;
    }

    @Override
    protected SubmissionType[] getActiveSubmissionTypes() {
        final SubmissionType[] types = { SubmissionType.COMPLETE, SubmissionType.P14PART, SubmissionType.P35PART };
        return types;
    }

    public void processRule() {
        if (getSubmissionType() == SubmissionType.P35PART) {
            if ("".equals(numberOfParts)) {
                raiseError(DocumentError.MANDATORY);
            }
        } else if (!"".equals(numberOfParts) && getSubmissionChannel() == SubmissionChannel.FBI) {
            raiseError(NOT_REQUIRED);
        }

        /*
         * The rules for paper allow a range of 000-999 for the number of parts.
         * The schema check uses a range of 001-999 so in the handleError method
         * I have suppressed the MINVALUE error and we need to do the check
         * here.
         */
        if (getSubmissionChannel() == SubmissionChannel.Paper) {
            try {
                final long numberOfPartsValue = Long.parseLong(numberOfParts);
                if (numberOfPartsValue < 0) {
                    raiseError(DocumentError.MINVALUE);
                }
            } catch (final NumberFormatException e) {
                // If the number was badly formatted then the schema checks
                // would have picked that up so no need to do anything here
            }
        }
    }

    @Override
    protected String[] getInvocationPoints() {
        return new String[] { "/IRenvelope/EndOfYearReturn" };
    }

    @Override
    protected ActiveTaxYear[] getApplicableTaxYears() {
        return TaxYear.allActive();
    }

    @Override
    protected void initializeValueBindings() {
        getValueBinder().add("/IRenvelope/EndOfYearReturn/NumberOfParts", "numberOfParts");
    }

    @Override
    protected void initializeEventBindings() {
        // do nothing
    }

    public void setNumberOfParts(final String numberOfParts) {
        this.numberOfParts = numberOfParts;
    }

    @Override
    protected void initializeErrorBindings() {
        getErrorBinder().add("/IRenvelope/EndOfYearReturn/NumberOfParts", "handleError");
    }

    @Override
    public boolean handleError(final DocumentError documentError) {
        if (documentError.getSubCode().equals(DocumentError.MINVALUE) && getSubmissionChannel() == SubmissionChannel.Paper) {
            // Paper allows 000-999 but the schema checks for 001-999 so trap
            // and suppress the MINVALUE error. We'll check it in the rule
            // instead.
            return true;
        } else {
            return super.handleError(documentError);
        }
    }
}
