/**
 * 
 */
package paye.eoy.bizrules;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import paye.multiyear.DateHelper;
import paye.multiyear.TaxYear;
import xmldoc.DocumentError;
import xmldoc.ElementReference;
import bizrules.AbstractBusinessRule;
import bizrules.binding.ElementBinding;
import bizrules.binding.ValueBinder;

/**
 * @author Rob
 */
public abstract class PeriodEndRule extends AbstractBusinessRule {

    public static final String INVALID_PERIOD_END = "invalid period end";

    private String periodEnd = "";

    private boolean wasPeriodEndTagEncountered = false;

    private final List<ElementBinding> binders = new ArrayList<ElementBinding>();

    private final TaxYear taxYear;

    private final String[] subcodes = { INVALID_PERIOD_END, DocumentError.FORMAT, DocumentError.BLANK, DocumentError.MANDATORY };

    private String collectedNamespace = "";

    public PeriodEndRule(final TaxYear taxYear) {
        this.taxYear = taxYear;

        setRuleName("PeriodEndRule");

        final String[] namespaces = getApplicableNamespaces();
        for (final String element : namespaces) {
            // bind a period end processor for each of the tax years
            final ValueBinder binder = new ValueBinder(element, this);
            binder.add("/IRenvelope/IRheader/PeriodEnd", "periodEnd");
            binder.add("/IRenvelope", "iREnvelope");
            binders.addAll(binder.getBindings());
        }
        getErrorBinder().add("/IRenvelope/IRheader/PeriodEnd", "handleError");
    }

    @Override
    public String[] getSubcodes() {
        return subcodes;
    }

    public void setPeriodEnd(final String value) {
        periodEnd = value;
        wasPeriodEndTagEncountered = true;
    }

    /**
     * We want to trap the actual namespace used in the submission so that we
     * can compare it against the PeriodEnd date to check that they correspond.
     * 
     * @param attributes
     */
    public void setIREnvelope(final ElementReference element) {
        this.collectedNamespace = element.getNamespaceURL();
    }

    @Override
    protected String[] getInvocationPoints() {
        return new String[] { "/IRenvelope" };
    }

    @Override
    public List<ElementBinding> getValueBindings() {
        return binders;
    }

    @Override
    protected String[] getApplicableNamespaces() {
        return TaxYear.getUniqueNamespaces();
    }

    public void processRule() {
        // Schema will take care of format, blank or missing
        if (wasPeriodEndTagEncountered && !"".equals(periodEnd)) {
            if (!collectedNamespace.equals(taxYear.getIREnvelopeNamespaceString())) {
                raiseError(INVALID_PERIOD_END, periodEnd);
            } else {
                try {
                    final Date periodEndDate = DateHelper.createDateFormat().parse(periodEnd);

                    if (taxYear != taxYearFromPeriodEndDate(periodEndDate)) {
                        raiseError(INVALID_PERIOD_END, periodEnd);
                    }
                } catch (final ParseException e) {
                    // ignore - the schema will have generated an error
                }
            }
        }
    }

    protected abstract TaxYear taxYearFromPeriodEndDate(Date periodEndDate);

    /**
     * PeriodEndRule is always active as we need it to look for the period end
     * so we can decide which other rules should be active.
     */
    @Override
    public boolean isActive() {
        return true;
    }

    @Override
    protected void initializeValueBindings() {
    }

    @Override
    protected void initializeEventBindings() {
    }
}
