/*
 * Created on 03-Jan-2006
 *
 * $Id: CombinedLessStatutoryPaymentsRule.java,v 1.3 2006/08/10 15:19:16 dec Exp $
 */
package paye.eoy.bizrules.consolidated;

import java.math.BigDecimal;

import paye.eoy.bizrules.helpers.BigDecimalUtil;
import xmldoc.DocumentError;
import bizrules.binding.ErrorBinder;
import bizrules.errors.MonetaryDiscrepancy;

/**
 * $Id: CombinedLessStatutoryPaymentsRule.java,v 1.1 2006/01/04 14:25:41 raustin
 * Exp $
 */
public final class CombinedLessStatutoryPaymentsRule extends AbstractConsolidatedRule {
    public static final String BAD_TOTAL = "bad total";

    private final String[] subcodes = { BAD_TOTAL, DocumentError.FORMAT, DocumentError.MANDATORY };

    @Override
    public String[] getSubcodes() {
        return subcodes;
    }

    @Override
    protected String[] getApplicableNamespaces() {
        return new String[] { "http://www.govtalk.gov.uk/taxation/EOY-consolidated/2",
                "http://www.govtalk.gov.uk/taxation/EOY-consolidated/3", "http://www.govtalk.gov.uk/taxation/EOY-consolidated/3.1" };
    }

    private BigDecimal combinedLessStatutoryRecovered = BigDecimalUtil.ZERO;

    private BigDecimal totalNICsTaxAndStudentLoan = BigDecimalUtil.ZERO;

    private BigDecimal netStatutoryPaymentsRecovered = BigDecimalUtil.ZERO;

    @Override
    protected final void initializeValueBindings() {

        getValueBinder().add("/IRenvelope/EndOfYearReturn/P35/Summary/CombinedAmounts/CombinedLessStatutoryRecovered",
                "combinedLessStatutoryRecovered");
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P35Amended/Summary/CombinedAmounts/CombinedLessStatutoryRecovered",
                "combinedLessStatutoryRecovered");

        getValueBinder().add("/IRenvelope/EndOfYearReturn/P35/Summary/CombinedAmounts/TotalNICsTaxAndStudentLoan",
                "totalNICsTaxAndStudentLoan");
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P35Amended/Summary/CombinedAmounts/TotalNICsTaxAndStudentLoan",
                "totalNICsTaxAndStudentLoan");

        getValueBinder().add("/IRenvelope/EndOfYearReturn/P35/Summary/CombinedAmounts/NetStatutoryPaymentsRecovered",
                "netStatutoryPaymentsRecovered");
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P35Amended/Summary/CombinedAmounts/NetStatutoryPaymentsRecovered",
                "netStatutoryPaymentsRecovered");

    }

    public void processRule() {
        final BigDecimal calculatedTotal = totalNICsTaxAndStudentLoan.subtract(netStatutoryPaymentsRecovered);

        if (calculatedTotal.compareTo(combinedLessStatutoryRecovered) != 0) {
            raiseQS3Error(BAD_TOTAL, new MonetaryDiscrepancy(calculatedTotal, combinedLessStatutoryRecovered));
        }
    }

    @Override
    protected String[] getInvocationPoints() {
        return new String[] { "/IRenvelope/EndOfYearReturn/P35Amended/Summary/CombinedAmounts",
                "/IRenvelope/EndOfYearReturn/P35/Summary/CombinedAmounts" };
    }

    @Override
    protected void initializeEventBindings() {
        // do nothing
    }

    public void setCombinedLessStatutoryRecovered(final String combinedLessStatutoryRecovered) {
        this.combinedLessStatutoryRecovered = BigDecimalUtil.toBigDecimal(combinedLessStatutoryRecovered);
    }

    public void setNetStatutoryPaymentsRecovered(final String netStatutoryPaymentsRecovered) {
        this.netStatutoryPaymentsRecovered = BigDecimalUtil.toBigDecimal(netStatutoryPaymentsRecovered);
    }

    public void setTotalNICsTaxAndStudentLoan(final String totalNICsTaxAndStudentLoan) {
        this.totalNICsTaxAndStudentLoan = BigDecimalUtil.toBigDecimal(totalNICsTaxAndStudentLoan);
    }

    @Override
    protected void initializeErrorBindings() {
        final ErrorBinder binder = getErrorBinder();
        binder.add("/IRenvelope/EndOfYearReturn/P35/Summary/CombinedAmounts/CombinedLessStatutoryRecovered", "handleError");
        binder.add("/IRenvelope/EndOfYearReturn/P35Amended/Summary/CombinedAmounts/CombinedLessStatutoryRecovered", "handleError");
    }
}
