package paye.eoy.bizrules.consolidated;

import java.math.BigDecimal;

import paye.eoy.bizrules.helpers.BigDecimalUtil;
import xmldoc.DocumentError;
import bizrules.binding.ErrorBinder;
import bizrules.errors.MonetaryDiscrepancy;

/**
 * @author Doug
 */
public final class NetTaxCreditsRule extends AbstractConsolidatedRule {

    public static final String _1 = "1";

    private BigDecimal taxCredit = BigDecimalUtil.ZERO;

    private BigDecimal taxCreditFunding = BigDecimalUtil.ZERO;

    private BigDecimal netTaxCredits = BigDecimalUtil.ZERO;

    private final String[] subcodes = { _1, DocumentError.FORMAT, DocumentError.MANDATORY, DocumentError.BLANK };

    @Override
    public String[] getSubcodes() {
        return subcodes;
    }

    @Override
    protected String[] getApplicableNamespaces() {
        return new String[] { "http://www.govtalk.gov.uk/taxation/EOY-consolidated/1",
                "http://www.govtalk.gov.uk/taxation/EOY-consolidated/1.5" };
    }

    @Override
    protected void initializeValueBindings() {
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P35Amended/Summary/CombinedAmounts/TaxCredit", "taxCredit");
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P35Amended/Summary/CombinedAmounts/NetTaxCredits", "netTaxCredits");
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P35Amended/Summary/CombinedAmounts/TaxCreditFunding", "taxCreditFunding");

        getValueBinder().add("/IRenvelope/EndOfYearReturn/P35/Summary/CombinedAmounts/TaxCredit", "taxCredit");
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P35/Summary/CombinedAmounts/NetTaxCredits", "netTaxCredits");
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P35/Summary/CombinedAmounts/TaxCreditFunding", "taxCreditFunding");
    }

    public void processRule() {

        final BigDecimal calculatedTotal = taxCredit.subtract(taxCreditFunding);

        if (netTaxCredits.compareTo(calculatedTotal) != 0) {
            raiseQS3Error(_1, new MonetaryDiscrepancy(calculatedTotal, netTaxCredits));
        }

    }

    @Override
    protected String[] getInvocationPoints() {
        return new String[] { "/IRenvelope/EndOfYearReturn/P35Amended", "/IRenvelope/EndOfYearReturn/P35" };
    }

    @Override
    protected void initializeEventBindings() {
        // do nothing
    }

    public void setNetTaxCredits(final String netTaxCredits) {
        this.netTaxCredits = BigDecimalUtil.toBigDecimal(netTaxCredits);
    }

    public void setTaxCredit(final String taxCredit) {
        this.taxCredit = BigDecimalUtil.toBigDecimal(taxCredit);
    }

    public void setTaxCreditFunding(final String taxCreditFunding) {
        this.taxCreditFunding = BigDecimalUtil.toBigDecimal(taxCreditFunding);
    }

    @Override
    protected void initializeErrorBindings() {
        final ErrorBinder binder = getErrorBinder();
        binder.add("/IRenvelope/EndOfYearReturn/P35/Summary/CombinedAmounts/NetTaxCredits", "handleError");
        binder.add("/IRenvelope/EndOfYearReturn/P35Amended/Summary/CombinedAmounts/NetTaxCredits", "handleError");
    }
}
